<?php
/**
 * Plugin Name: Subscription Tracker From Web321
 * Plugin URI: https://web321.co/our-plugins/w321_subscription_tracker
 * Description: Track paid plugin/theme subscriptions, renewal dates, and recurring WordPress costs.
 * Version: 0.1.2
 * Requires at least: 6.0
 * Requires PHP: 8.0
 * Author: Web321 Marketing Ltd.
 * Author URI: https://web321.co
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: wp-subscription-tracker
 */

if (! defined('ABSPATH')) {
    exit;
}

define('WPST_VERSION', '0.1.0');
define('WPST_PLUGIN_FILE', __FILE__);
define('WPST_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('WPST_PLUGIN_URL', plugin_dir_url(__FILE__));

require_once WPST_PLUGIN_DIR . 'includes/class-wpst-activator.php';
require_once WPST_PLUGIN_DIR . 'includes/class-wpst-deactivator.php';
require_once WPST_PLUGIN_DIR . 'includes/class-wpst-encryption.php';
require_once WPST_PLUGIN_DIR . 'includes/class-wpst-affiliates.php';
require_once WPST_PLUGIN_DIR . 'includes/class-wpst-detector.php';
require_once WPST_PLUGIN_DIR . 'includes/class-wpst-export.php';
require_once WPST_PLUGIN_DIR . 'includes/class-wpst-history.php';
require_once WPST_PLUGIN_DIR . 'includes/class-wpst-notifications.php';
require_once WPST_PLUGIN_DIR . 'includes/class-wpst-pricing-sync.php';
require_once WPST_PLUGIN_DIR . 'includes/class-wpst-admin.php';
require_once WPST_PLUGIN_DIR . 'includes/class-wpst-inventory-admin.php';

register_activation_hook(__FILE__, ['WPST_Activator', 'activate']);
register_deactivation_hook(__FILE__, ['WPST_Deactivator', 'deactivate']);


function wpst_maybe_upgrade_schema(): void
{
    $storedVersion = (string) get_option('wpst_schema_version', '');
    if ($storedVersion === WPST_VERSION) {
        return;
    }

    WPST_Activator::activate();
    update_option('wpst_schema_version', WPST_VERSION);
}

function wpst_run(): void
{
    load_plugin_textdomain('wp-subscription-tracker', false, dirname(plugin_basename(__FILE__)) . '/languages');

    wpst_maybe_upgrade_schema();

    $admin = new WPST_Admin();
    $admin->init();

    $inventoryAdmin = new WPST_Inventory_Admin();
    $inventoryAdmin->init();

    $notifications = new WPST_Notifications();
    $notifications->init();

    $history = new WPST_History();
    $history->init();

    $pricingSync = new WPST_Pricing_Sync();
    $pricingSync->init();

    $affiliates = new WPST_Affiliates();
    $affiliates->init();

    add_action('rest_api_init', 'wpst_register_rest_routes');

    add_filter('plugin_action_links_' . plugin_basename(__FILE__), 'wpst_plugin_action_links');
    add_filter('plugin_row_meta', 'wpst_plugin_row_meta', 10, 2);
}
add_action('plugins_loaded', 'wpst_run');

function wpst_register_rest_routes(): void
{
    register_rest_route('wp-subscription-tracker/v1', '/summary', [
        'methods'             => 'GET',
        'permission_callback' => static function (WP_REST_Request $request) {
            if (current_user_can('manage_options')) {
                return true;
            }

            $token = sanitize_text_field((string) $request->get_param('token'));
            if ($token === '') {
                return false;
            }

            $storedToken = (string) get_option('wpst_site_token', '');

            return $storedToken !== '' && hash_equals($storedToken, $token);
        },
        'callback'            => static function (WP_REST_Request $request) {
            global $wpdb;

            $table = $wpdb->prefix . 'subscription_tracker';
            $rows  = $wpdb->get_results($wpdb->prepare("SELECT id, item_name, item_type, vendor, cost, currency, billing_cycle, renewal_date, auto_renew, status FROM {$table} LIMIT %d", 1000000), ARRAY_A);

            $monthly = 0.0;
            $annual  = 0.0;
            $active  = 0;
            $nextRenewal = null;

            foreach ($rows as $row) {
                $cost  = (float) $row['cost'];
                $cycle = (string) $row['billing_cycle'];

                if ($row['status'] === 'active') {
                    $active++;
                }

                if ($cycle === 'monthly') {
                    $monthly += $cost;
                    $annual += $cost * 12;
                } elseif ($cycle === 'annually') {
                    $monthly += ($cost / 12);
                    $annual += $cost;
                }

                if (! empty($row['renewal_date']) && ($nextRenewal === null || $row['renewal_date'] < $nextRenewal['renewal_date'])) {
                    $nextRenewal = [
                        'id'           => (int) $row['id'],
                        'item_name'    => (string) $row['item_name'],
                        'item_type'    => (string) $row['item_type'],
                        'vendor'       => (string) $row['vendor'],
                        'renewal_date' => (string) $row['renewal_date'],
                        'cost'         => round((float) $row['cost'], 2),
                        'currency'     => (string) $row['currency'],
                    ];
                }
            }

            $response = [
                'total_monthly'      => round($monthly, 2),
                'total_annual'       => round($annual, 2),
                'active_count'       => $active,
                'total_subscriptions'=> count($rows),
                'next_renewal'       => $nextRenewal,
                'generated_at'       => gmdate('c'),
            ];

            if ((bool) $request->get_param('include_items')) {
                $response['items'] = array_map(static function (array $row): array {
                    return [
                        'id'           => (int) $row['id'],
                        'item_name'    => (string) $row['item_name'],
                        'item_type'    => (string) $row['item_type'],
                        'vendor'       => (string) $row['vendor'],
                        'cost'         => round((float) $row['cost'], 2),
                        'currency'     => (string) $row['currency'],
                        'billing_cycle'=> (string) $row['billing_cycle'],
                        'renewal_date' => (string) $row['renewal_date'],
                        'auto_renew'   => (int) $row['auto_renew'] === 1,
                        'status'       => (string) $row['status'],
                    ];
                }, $rows);
            }

            return rest_ensure_response($response);
        },
    ]);
}

function wpst_plugin_action_links(array $links): array
{
    $url = admin_url('tools.php?page=wpst-subscription-tracker');
    array_unshift($links, sprintf('<a href="%s">%s</a>', esc_url($url), esc_html__('Settings', 'wp-subscription-tracker')));

    return $links;
}

function wpst_plugin_row_meta(array $links, string $file): array
{
    if ($file !== plugin_basename(__FILE__)) {
        return $links;
    }

    $links[] = '<a href="https://www.paypal.com/paypalme/web321co/10" target="_blank" rel="noopener noreferrer">' . esc_html__('Donate', 'wp-subscription-tracker') . '</a>';

    return $links;
}
