<?php

if (! defined('ABSPATH')) {
    exit;
}

class WPST_Pricing_Sync
{
    private string $api_base;
    private int $max_retries = 3;
    private int $retry_backoff_ms = 500;
    private int $min_request_interval_seconds = 30;

    public function __construct()
    {
        $configuredBase = (string) get_option('wpst_pricing_api_base', 'https://shop.web321.co/v1/license_tracker/');
        $this->api_base = rtrim($configuredBase, '/');
    }

    public function init(): void
    {
        add_action('wpst_daily_pricing_sync', [$this, 'sync_all_pricing']);
    }

    public function sync_all_pricing(): void
    {
        global $wpdb;

        $table = $wpdb->prefix . 'subscription_tracker';
        $rows  = $wpdb->get_results($wpdb->prepare("SELECT id, item_slug, item_name, billing_cycle, cost, market_price FROM {$table} WHERE status IN ('active', 'trial') LIMIT %d", 1000000), ARRAY_A);
        if (! is_array($rows) || empty($rows)) {
            update_option('wpst_last_pricing_sync', current_time('mysql'), false);
            return;
        }

        $slugs = [];
        foreach ($rows as $row) {
            $slug = sanitize_title((string) ($row['item_slug'] ?? ''));
            if ($slug === '') {
                $slug = sanitize_title((string) ($row['item_name'] ?? ''));
            }

            if ($slug !== '') {
                $slugs[] = $slug;
            }
        }

        $slugs = array_values(array_unique($slugs));
        if (empty($slugs)) {
            update_option('wpst_last_pricing_sync', current_time('mysql'), false);
            return;
        }

        $pricingBySlug = $this->fetch_batch_pricing($slugs);

        foreach ($rows as $row) {
            $slug = sanitize_title((string) ($row['item_slug'] ?? ''));
            if ($slug === '') {
                $slug = sanitize_title((string) ($row['item_name'] ?? ''));
            }

            if ($slug === '' || ! isset($pricingBySlug[$slug])) {
                continue;
            }

            $product = $pricingBySlug[$slug];
            $newPrice = $this->extract_price_for_cycle((array) ($product['pricing'] ?? []), (string) ($row['billing_cycle'] ?? 'annually'));
            if ($newPrice === null) {
                continue;
            }

            $currentCost   = (float) ($row['cost'] ?? 0);
            $currentMarket = isset($row['market_price']) && $row['market_price'] !== null ? (float) $row['market_price'] : null;
            $isOverridden  = $currentMarket !== null && abs($currentCost - $currentMarket) > 0.0001;

            $updateData = [
                'market_price'         => $newPrice,
                'pricing_source_url'   => esc_url_raw((string) ($product['url'] ?? '')),
                'pricing_last_synced'  => current_time('mysql'),
                'updated_at'           => current_time('mysql'),
            ];

            if (! $isOverridden) {
                $updateData['cost'] = $newPrice;
            }

            $wpdb->update($table, $updateData, ['id' => (int) $row['id']]);

            if ($currentMarket !== null && abs($currentMarket - $newPrice) > 0.0001) {
                $this->log_price_change((int) $row['id'], $currentMarket, $newPrice);
            }
        }

        update_option('wpst_last_pricing_sync', current_time('mysql'), false);
    }

    /**
     * @param array<int, string> $slugs
     * @return array<string, array<string, mixed>>
     */
    private function fetch_batch_pricing(array $slugs): array
    {
        $this->wait_for_rate_limit_window();

        $requestArgs = [
            'headers' => [
                'Authorization' => 'Bearer ' . $this->get_api_key(),
                'Content-Type'  => 'application/json',
            ],
            'body'    => wp_json_encode([
                'slugs'      => array_values($slugs),
                'site_token' => sanitize_text_field((string) get_option('wpst_site_token', '')),
            ]),
            'timeout' => 20,
        ];

        $response = null;
        for ($attempt = 1; $attempt <= $this->max_retries; $attempt++) {
            $response = wp_remote_post($this->api_base . '/batch', $requestArgs);

            if (is_wp_error($response)) {
                $this->sleep_before_retry($attempt);
                continue;
            }

            $status = (int) wp_remote_retrieve_response_code($response);
            if ($status >= 200 && $status < 300) {
                break;
            }

            if ($status !== 429 && $status < 500) {
                return [];
            }

            $this->sleep_before_retry($attempt);
        }

        if (is_wp_error($response)) {
            return [];
        }

        $status = (int) wp_remote_retrieve_response_code($response);
        if ($status < 200 || $status >= 300) {
            return [];
        }

        set_transient('wpst_pricing_last_request_ts', time(), $this->min_request_interval_seconds);

        $payload = json_decode((string) wp_remote_retrieve_body($response), true);
        if (! is_array($payload)) {
            return [];
        }

        $products = $payload['data'] ?? $payload;
        if (! is_array($products)) {
            return [];
        }

        $map = [];
        foreach ($products as $key => $product) {
            if (! is_array($product)) {
                continue;
            }

            $slug = sanitize_title((string) ($product['slug'] ?? (is_string($key) ? $key : '')));
            if ($slug === '') {
                continue;
            }

            $map[$slug] = $product;
        }

        return $map;
    }

    private function wait_for_rate_limit_window(): void
    {
        $lastRequest = get_transient('wpst_pricing_last_request_ts');
        if (! is_numeric($lastRequest)) {
            return;
        }

        $elapsed = time() - (int) $lastRequest;
        if ($elapsed >= $this->min_request_interval_seconds) {
            return;
        }

        $waitSeconds = $this->min_request_interval_seconds - $elapsed;
        if ($waitSeconds > 0) {
            sleep($waitSeconds);
        }
    }

    private function sleep_before_retry(int $attempt): void
    {
        if ($attempt >= $this->max_retries) {
            return;
        }

        $backoffMs = $this->retry_backoff_ms * (2 ** ($attempt - 1));
        usleep($backoffMs * 1000);
    }

    private function get_api_key(): string
    {
        return sanitize_text_field((string) get_option('wpst_pricing_api_key', ''));
    }

    /**
     * @param array<string, mixed> $pricing
     */
    private function extract_price_for_cycle(array $pricing, string $billingCycle): ?float
    {
        $cycleMap = [
            'monthly'  => 'monthly',
            'annually' => 'annual',
            'lifetime' => 'lifetime',
            'one-time' => 'annual',
        ];

        $normalizedCycle = $cycleMap[$billingCycle] ?? 'annual';

        if (isset($pricing[$normalizedCycle])) {
            return (float) $pricing[$normalizedCycle];
        }

        $cycles = isset($pricing['billing_cycles']) && is_array($pricing['billing_cycles']) ? $pricing['billing_cycles'] : [];
        if (isset($cycles[$normalizedCycle]['price'])) {
            return (float) $cycles[$normalizedCycle]['price'];
        }

        if (isset($pricing['annual'])) {
            return (float) $pricing['annual'];
        }

        return null;
    }

    private function log_price_change(int $subscriptionId, float $oldPrice, float $newPrice): void
    {
        global $wpdb;

        $table = $wpdb->prefix . 'subscription_price_changes';
        $percent = $oldPrice > 0 ? (($newPrice - $oldPrice) / $oldPrice) * 100 : 0;

        $wpdb->insert($table, [
            'subscription_id' => $subscriptionId,
            'old_price'       => $oldPrice,
            'new_price'       => $newPrice,
            'change_percent'  => round($percent, 2),
            'detected_at'     => current_time('mysql'),
            'notified'        => 0,
        ]);
    }

    /**
     * @return array<int, array<string, mixed>>
     */
    public function get_price_alerts(): array
    {
        global $wpdb;

        $subscriptionsTable = $wpdb->prefix . 'subscription_tracker';
        $changesTable       = $wpdb->prefix . 'subscription_price_changes';

        $sql = "SELECT s.id, s.item_name, s.market_price, s.cost, c.old_price, c.new_price, c.change_percent, c.detected_at\n            FROM {$subscriptionsTable} s\n            INNER JOIN {$changesTable} c ON c.subscription_id = s.id\n            WHERE c.notified = 0\n            ORDER BY c.detected_at DESC\n            LIMIT %d";

        return (array) $wpdb->get_results($wpdb->prepare($sql, 10), ARRAY_A);
    }
}
