<?php

if (! defined('ABSPATH')) {
    exit;
}

class WPST_Notifications
{
    public function init(): void
    {
        add_action('wpst_daily_check', [$this, 'run_daily_check']);
    }

    public function run_daily_check(): void
    {
        global $wpdb;
        $table = $wpdb->prefix . 'subscription_tracker';

        $rows = $wpdb->get_results($wpdb->prepare("SELECT id, item_name, vendor, cost, currency, renewal_date, status, affiliate_url FROM {$table} WHERE status = %s AND renewal_date IS NOT NULL LIMIT %d", 'active', 1000000), ARRAY_A);

        $intervals = array_values(array_unique(array_map('intval', (array) get_option('wpst_reminder_intervals', ['30', '7', '1', '0']))));
        $sent      = get_option('wpst_sent_reminders', []);
        $to        = get_option('wpst_notification_email', get_option('admin_email'));

        if (! is_array($sent)) {
            $sent = [];
        }

        foreach ($rows as $row) {
            $days = (int) floor((strtotime((string) $row['renewal_date']) - current_time('timestamp')) / DAY_IN_SECONDS);
            if (! in_array($days, $intervals, true)) {
                continue;
            }

            $id = (int) $row['id'];
            if (isset($sent[$id][(string) $days])) {
                continue;
            }

            $subject = sprintf(__('Renewal reminder: %s', 'wp-subscription-tracker'), $row['item_name']);
            $body    = $this->render_email_template($row, $days);

            $headers = ['Content-Type: text/html; charset=UTF-8'];
            wp_mail($to, $subject, $body, $headers);

            if (! isset($sent[$id])) {
                $sent[$id] = [];
            }
            $sent[$id][(string) $days] = current_time('mysql');
        }

        $sent = $this->cleanup_stale_reminders($sent);

        update_option('wpst_sent_reminders', $sent, false);
    }

    /**
     * @param array<string, mixed> $row
     */
    private function render_email_template(array $row, int $days): string
    {
        $affiliateEnabled = get_option('wpst_affiliate_disabled', '0') !== '1';
        $affiliateUrl     = $affiliateEnabled ? esc_url((string) ($row['affiliate_url'] ?? '')) : '';

        $templateData = [
            'item_name'       => (string) ($row['item_name'] ?? ''),
            'vendor'          => (string) ($row['vendor'] ?? ''),
            'renewal_date'    => (string) ($row['renewal_date'] ?? ''),
            'days_until'      => $days,
            'formatted_price' => number_format((float) ($row['cost'] ?? 0), 2) . ' ' . (string) ($row['currency'] ?? 'USD'),
            'affiliate_url'   => $affiliateUrl,
        ];

        ob_start();
        include WPST_PLUGIN_DIR . 'templates/email-reminder.php';
        $content = ob_get_clean();

        if (! is_string($content) || $content === '') {
            return '';
        }

        return $content;
    }

    /**
     * @param array<int|string, mixed> $sent
     * @return array<int|string, mixed>
     */
    private function cleanup_stale_reminders(array $sent): array
    {
        $monthKey        = gmdate('Y-m');
        $lastCleanupKey  = (string) get_option('wpst_reminder_cleanup_month', '');
        if ($monthKey === $lastCleanupKey) {
            return $sent;
        }

        $threshold = strtotime('-12 months', current_time('timestamp'));

        foreach ($sent as $subscriptionId => $entries) {
            if (! is_array($entries)) {
                unset($sent[$subscriptionId]);
                continue;
            }

            foreach ($entries as $interval => $stamp) {
                $stampTs = is_string($stamp) ? strtotime($stamp) : false;
                if (! $stampTs || $stampTs < $threshold) {
                    unset($sent[$subscriptionId][$interval]);
                }
            }

            if (empty($sent[$subscriptionId])) {
                unset($sent[$subscriptionId]);
            }
        }

        update_option('wpst_reminder_cleanup_month', $monthKey, false);

        return $sent;
    }
}
