<?php

if (! defined('ABSPATH')) {
    exit;
}

class WPST_Inventory_Admin
{
    public function init(): void
    {
        add_action('admin_enqueue_scripts', [$this, 'enqueue']);
    }

    public function enqueue(string $hook): void
    {
        if (! in_array($hook, ['plugins.php', 'themes.php'], true)) {
            return;
        }

        if (! current_user_can('manage_options')) {
            return;
        }

        wp_enqueue_style('wpst-inventory', WPST_PLUGIN_URL . 'admin/css/wpst-inventory.css', [], WPST_VERSION);
        wp_enqueue_script('wpst-inventory', WPST_PLUGIN_URL . 'admin/js/wpst-inventory.js', ['jquery'], WPST_VERSION, true);

        wp_localize_script('wpst-inventory', 'wpstInventory', [
            'ajaxUrl'       => admin_url('admin-ajax.php'),
            'nonce'         => wp_create_nonce('wpst_ajax'),
            'screen'        => $hook,
            'subscriptions' => $this->get_inventory_subscriptions(),
            'pluginMeta'    => $hook === 'plugins.php' ? $this->get_plugin_metadata() : [],
            'defaults'      => [
                'currency'  => (string) get_option('wpst_default_currency', 'CAD'),
                'currencies'=> ['CAD', 'USD', 'EUR', 'GBP', 'AUD'],
            ],
            'i18n'          => [
                'track'                 => __('Track', 'wp-subscription-tracker'),
                'modalTitleNew'         => __('Track Subscription', 'wp-subscription-tracker'),
                'modalTitleEdit'        => __('Update Subscription', 'wp-subscription-tracker'),
                'save'                  => __('Save Subscription Info', 'wp-subscription-tracker'),
                'saving'                => __('Saving…', 'wp-subscription-tracker'),
                'cancel'                => __('Cancel', 'wp-subscription-tracker'),
                'expiringSoon'          => __('Expires in %d day(s): %s', 'wp-subscription-tracker'),
                'promotionAvailable'    => __('Promotion noted. Review offer details.', 'wp-subscription-tracker'),
                'noAction'              => __('No urgent updates.', 'wp-subscription-tracker'),
                'saveFailed'            => __('Unable to save subscription.', 'wp-subscription-tracker'),
                'saveSuccess'           => __('Subscription saved.', 'wp-subscription-tracker'),
                'vendorHandling'        => __('Vendor', 'wp-subscription-tracker'),
                'managementHandling'    => __('Managed by website company', 'wp-subscription-tracker'),
            ],
        ]);
    }

    /**
     * @return array<string, array{vendor_name: string, vendor_url: string}>
     */
    private function get_plugin_metadata(): array
    {
        if (! function_exists('get_plugins')) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }

        $plugins = get_plugins();
        if (! is_array($plugins) || empty($plugins)) {
            return [];
        }

        $meta = [];
        foreach ($plugins as $pluginFile => $pluginData) {
            $pluginFile = (string) $pluginFile;
            if ($pluginFile === '') {
                continue;
            }

            $slug = sanitize_title(dirname($pluginFile));
            if ($slug === '.' || $slug === '') {
                $slug = sanitize_title(basename($pluginFile, '.php'));
            }

            if ($slug === '') {
                continue;
            }

            $vendorName = trim(wp_strip_all_tags((string) ($pluginData['AuthorName'] ?? $pluginData['Author'] ?? '')));
            $vendorUrl  = esc_url_raw((string) ($pluginData['AuthorURI'] ?? ''));

            if ($vendorUrl === '') {
                $vendorUrl = esc_url_raw((string) ($pluginData['PluginURI'] ?? ''));
            }

            $meta[$slug] = [
                'vendor_name' => $vendorName,
                'vendor_url'  => $vendorUrl,
            ];
        }

        return $meta;
    }

    /**
     * @return array<string, array<string, mixed>>
     */
    private function get_inventory_subscriptions(): array
    {
        global $wpdb;

        $table = $wpdb->prefix . 'subscription_tracker';
        $rows  = $wpdb->get_results($wpdb->prepare("SELECT * FROM {$table} WHERE item_type IN (%s, %s) ORDER BY updated_at DESC LIMIT %d", 'plugin', 'theme', 5000), ARRAY_A);

        if (! is_array($rows) || empty($rows)) {
            return [];
        }

        $items = [];
        foreach ($rows as $row) {
            $itemType = sanitize_key((string) ($row['item_type'] ?? ''));
            $slug     = sanitize_title((string) ($row['item_slug'] ?? ''));

            if ($itemType === '' || $slug === '') {
                continue;
            }

            $key = $itemType . ':' . $slug;
            if (isset($items[$key])) {
                continue;
            }

            $items[$key] = [
                'id'           => (int) $row['id'],
                'item_type'    => $itemType,
                'item_slug'    => $slug,
                'item_name'    => (string) $row['item_name'],
                'vendor'            => (string) $row['vendor'],
                'vendor_url'        => (string) ($row['vendor_url'] ?? ''),
                'handled_by'        => (string) ($row['handled_by'] ?? 'vendor'),
                'management_company'=> (string) ($row['management_company'] ?? ''),
                'cost'         => (float) $row['cost'],
                'currency'     => (string) $row['currency'],
                'billing_cycle'=> (string) $row['billing_cycle'],
                'renewal_date' => (string) ($row['renewal_date'] ?? ''),
                'status'       => (string) $row['status'],
                'notes'        => (string) ($row['notes'] ?? ''),
            ];
        }

        return $items;
    }
}
