<?php

if (! defined('ABSPATH')) {
    exit;
}

class WPST_Export
{
    /**
     * @return array<int, array<string, string>>
     */
    private static function get_rows(): array
    {
        global $wpdb;
        $table = $wpdb->prefix . 'subscription_tracker';

        return $wpdb->get_results($wpdb->prepare("SELECT id, item_name, vendor, cost, currency, billing_cycle, renewal_date, status, affiliate_url FROM {$table} ORDER BY item_name ASC LIMIT %d", 1000000), ARRAY_A);
    }

    /**
     * @param resource $output
     * @param array<int, array<string, mixed>> $rows
     */
    public static function write_csv_rows($output, array $rows): void
    {
        fputcsv($output, ['Name', 'Vendor', 'Cost', 'Currency', 'Cycle', 'Renewal Date', 'Status']);

        foreach ($rows as $row) {
            fputcsv($output, [
                (string) ($row['item_name'] ?? ''),
                (string) ($row['vendor'] ?? ''),
                (string) ($row['cost'] ?? ''),
                (string) ($row['currency'] ?? ''),
                (string) ($row['billing_cycle'] ?? ''),
                (string) ($row['renewal_date'] ?? ''),
                (string) ($row['status'] ?? ''),
            ]);
        }
    }

    public static function download(): void
    {
        if (! current_user_can('manage_options')) {
            wp_die(esc_html__('Insufficient permissions.', 'wp-subscription-tracker'));
        }

        check_admin_referer('wpst_export_csv');
        $rows = self::get_rows();

        nocache_headers();
        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename=wp-subscription-tracker.csv');

        $output = fopen('php://output', 'w');
        if (! $output) {
            exit;
        }

        self::write_csv_rows($output, $rows);

        fclose($output);
        exit;
    }

    public static function download_ical(): void
    {
        if (! current_user_can('manage_options')) {
            wp_die(esc_html__('Insufficient permissions.', 'wp-subscription-tracker'));
        }

        check_admin_referer('wpst_export_ical');

        nocache_headers();
        header('Content-Type: text/calendar; charset=utf-8');
        header('Content-Disposition: attachment; filename=wp-subscription-renewals.ics');

        echo self::build_ical(self::get_rows());
        exit;
    }

    /**
     * @param array<int, array<string, string>> $rows
     */
    public static function build_ical(array $rows): string
    {
        $lines = [
            'BEGIN:VCALENDAR',
            'VERSION:2.0',
            'PRODID:-//WP Subscription Tracker//EN',
            'CALSCALE:GREGORIAN',
            'METHOD:PUBLISH',
            'X-WR-CALNAME:WP Subscription Renewals',
        ];

        $stamp = gmdate('Ymd\THis\Z');
        $site  = wp_parse_url(home_url(), PHP_URL_HOST);

        foreach ($rows as $row) {
            if (empty($row['renewal_date'])) {
                continue;
            }

            $eventDate = gmdate('Ymd', strtotime((string) $row['renewal_date']));
            if ($eventDate === '19700101') {
                continue;
            }

            $title       = 'Renewal: ' . (string) $row['item_name'];
            $description = sprintf(
                'Vendor: %s\\nCost: %s %s\\nBilling Cycle: %s\\nStatus: %s',
                (string) $row['vendor'],
                number_format((float) $row['cost'], 2, '.', ''),
                (string) $row['currency'],
                (string) $row['billing_cycle'],
                (string) $row['status']
            );

            $lines[] = 'BEGIN:VEVENT';
            $lines[] = 'UID:wpst-' . (int) $row['id'] . '-' . $site;
            $lines[] = 'DTSTAMP:' . $stamp;
            $lines[] = 'DTSTART;VALUE=DATE:' . $eventDate;
            $lines[] = 'SUMMARY:' . self::escape_ical_text($title);
            $lines[] = 'DESCRIPTION:' . self::escape_ical_text($description);

            if (! empty($row['affiliate_url'])) {
                $lines[] = 'URL:' . self::escape_ical_text((string) $row['affiliate_url']);
            }

            $lines[] = 'END:VEVENT';
        }

        $lines[] = 'END:VCALENDAR';

        return implode("\r\n", $lines) . "\r\n";
    }

    private static function escape_ical_text(string $text): string
    {
        return str_replace(
            ["\\", ';', ',', "\r", "\n"],
            ['\\\\', '\\;', '\\,', '', '\\n'],
            $text
        );
    }
}
