<?php

if (! defined('ABSPATH')) {
    exit;
}

class WPST_Detector
{
    /**
     * @return array<int, array<string, mixed>>
     */
    public function scan(): array
    {
        if (! function_exists('get_plugins')) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }

        $known             = $this->get_known_plugins();
        $found             = [];
        $updateSignals     = $this->get_update_transient_signals();
        $licenseOptionKeys = $this->get_license_option_keys();

        foreach (get_plugins() as $file => $plugin) {
            $slug = strtolower($file);

            if (isset($known[$slug])) {
                $found[] = [
                    'item_type'     => 'plugin',
                    'item_slug'     => $slug,
                    'item_name'     => $known[$slug]['name'] ?? $plugin['Name'],
                    'vendor'        => $known[$slug]['vendor'] ?? $plugin['AuthorName'],
                    'cost'          => (float) ($known[$slug]['estimated_cost'] ?? 0),
                    'currency'      => $known[$slug]['currency'] ?? 'USD',
                    'billing_cycle' => $known[$slug]['billing_cycle'] ?? 'annually',
                    'is_detected'   => 1,
                    'status'        => 'active',
                ];
                continue;
            }

            $name         = (string) ($plugin['Name'] ?? '');
            $updateUri    = (string) ($plugin['UpdateURI'] ?? '');
            $slugSignals  = $this->extract_slug_signals($slug);
            $needsReview  = false;
            $isPaidSignal = false;

            if ($this->name_looks_premium($name)) {
                $isPaidSignal = true;
                $needsReview  = true;
            }

            if ($updateUri !== '' && ! str_contains($updateUri, 'wordpress.org')) {
                $isPaidSignal = true;
                $needsReview  = false;
            }

            if ($this->array_has_any_key($updateSignals, $slugSignals)) {
                $isPaidSignal = true;
                $needsReview  = false;
            }

            if ($this->array_has_any_key($licenseOptionKeys, $slugSignals)) {
                $isPaidSignal = true;
                $needsReview  = false;
            }

            if ($isPaidSignal) {
                $found[] = [
                    'item_type'     => 'plugin',
                    'item_slug'     => $slug,
                    'item_name'     => $name,
                    'vendor'        => (string) ($plugin['AuthorName'] ?? ''),
                    'cost'          => 0.00,
                    'currency'      => get_option('wpst_default_currency', 'CAD'),
                    'billing_cycle' => 'annually',
                    'is_detected'   => 1,
                    'status'        => $needsReview ? 'needs_review' : 'active',
                ];
            }
        }

        foreach (wp_get_themes() as $slug => $theme) {
            $name = (string) $theme->get('Name');

            if (! $this->name_looks_premium($name)) {
                continue;
            }

            $found[] = [
                'item_type'     => 'theme',
                'item_slug'     => (string) $slug,
                'item_name'     => $name,
                'vendor'        => (string) $theme->get('Author'),
                'cost'          => 0.00,
                'currency'      => get_option('wpst_default_currency', 'CAD'),
                'billing_cycle' => 'annually',
                'is_detected'   => 1,
                'status'        => 'needs_review',
            ];
        }

        return $found;
    }

    /**
     * @return array<string, array<string, mixed>>
     */
    private function get_known_plugins(): array
    {
        $path = WPST_PLUGIN_DIR . 'data/known-plugins.json';
        if (! file_exists($path)) {
            return [];
        }

        $json = file_get_contents($path);
        if (! is_string($json) || $json === '') {
            return [];
        }

        $decoded = json_decode($json, true);

        return is_array($decoded) ? $decoded : [];
    }

    /**
     * @return array<string, bool>
     */
    private function get_update_transient_signals(): array
    {
        $signals  = [];
        $update   = get_site_transient('update_plugins');
        $response = isset($update->response) && is_array($update->response) ? $update->response : [];

        foreach ($response as $pluginFile => $data) {
            $pluginFile = strtolower((string) $pluginFile);
            foreach ($this->extract_slug_signals($pluginFile) as $signal) {
                $signals[$signal] = true;
            }

            $package = isset($data->package) ? (string) $data->package : '';
            if ($package !== '' && ! str_contains($package, 'downloads.wordpress.org')) {
                foreach ($this->extract_slug_signals($package) as $signal) {
                    $signals[$signal] = true;
                }
            }
        }

        return $signals;
    }

    /**
     * @return array<string, bool>
     */
    private function get_license_option_keys(): array
    {
        global $wpdb;

        $signals = [];
        $table   = $wpdb->options;
        $rows    = $wpdb->get_col("SELECT option_name FROM {$table} WHERE option_name LIKE '%license%' OR option_name LIKE '%licence%'"); // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared

        foreach ($rows as $name) {
            foreach ($this->extract_slug_signals((string) $name) as $signal) {
                $signals[$signal] = true;
            }
        }

        return $signals;
    }

    /**
     * @return array<int, string>
     */
    private function extract_slug_signals(string $value): array
    {
        $value = strtolower($value);
        $value = preg_replace('/[^a-z0-9_\/-]+/', '-', $value);
        if (! is_string($value) || $value === '') {
            return [];
        }

        $parts   = preg_split('/[\/_-]+/', $value);
        $signals = [];

        if (is_array($parts)) {
            foreach ($parts as $part) {
                $part = trim($part);
                if ($part !== '' && strlen($part) > 2) {
                    $signals[] = $part;
                }
            }
        }

        return array_values(array_unique($signals));
    }

    /**
     * @param array<string, bool> $haystack
     * @param array<int, string>  $needles
     */
    private function array_has_any_key(array $haystack, array $needles): bool
    {
        foreach ($needles as $needle) {
            if (isset($haystack[$needle])) {
                return true;
            }
        }

        return false;
    }

    private function name_looks_premium(string $name): bool
    {
        $name    = strtolower($name);
        $signals = [' pro', ' premium', ' elite', ' agency', ' business'];

        foreach ($signals as $signal) {
            if (str_contains($name, $signal)) {
                return true;
            }
        }

        return false;
    }
}
