<?php

if (! defined('ABSPATH')) {
    exit;
}

class WPST_Affiliates
{
    public function init(): void
    {
        add_filter('cron_schedules', [$this, 'add_weekly_schedule']);
        add_action('wpst_weekly_aff_sync', [$this, 'sync_affiliate_feed']);
    }

    /**
     * @param array<string, array<string, mixed>> $schedules
     * @return array<string, array<string, mixed>>
     */
    public function add_weekly_schedule(array $schedules): array
    {
        if (! isset($schedules['weekly'])) {
            $schedules['weekly'] = [
                'interval' => WEEK_IN_SECONDS,
                'display'  => __('Once Weekly', 'wp-subscription-tracker'),
            ];
        }

        return $schedules;
    }

    public function sync_affiliate_feed(): void
    {
        if (get_option('wpst_affiliate_disabled', '0') === '1') {
            delete_transient('wpst_affiliate_links');
            return;
        }

        $response = wp_remote_get('https://web321.co/aff_links.xml', ['timeout' => 15, 'sslverify' => true]);
        if (is_wp_error($response)) {
            return;
        }

        $xml = wp_remote_retrieve_body($response);
        if (! is_string($xml) || $xml === '') {
            return;
        }

        $links = $this->parse_affiliate_xml($xml);
        if ($links === []) {
            return;
        }

        set_transient('wpst_affiliate_links', $links, WEEK_IN_SECONDS);
        $this->apply_affiliate_links($links);
    }

    /**
     * @return array<string, array<string, string>>
     */
    public function parse_affiliate_xml(string $xml): array
    {
        if (trim($xml) === '') {
            return [];
        }

        if (preg_match('/<!DOCTYPE/i', $xml)) {
            return [];
        }

        libxml_use_internal_errors(true);
        $parsed = simplexml_load_string($xml, 'SimpleXMLElement', LIBXML_NONET | LIBXML_NOCDATA | LIBXML_NOERROR | LIBXML_NOWARNING);
        if ($parsed === false) {
            return [];
        }

        $links = $this->parse_link_nodes($parsed);
        if ($links !== []) {
            return $links;
        }

        $links = $this->parse_product_nodes($parsed);

        return $links;
    }

    /**
     * @return array<string, array<string, string>>
     */
    private function parse_link_nodes(SimpleXMLElement $parsed): array
    {
        $links = [];

        foreach ($parsed->link as $entry) {
            $slug = sanitize_title((string) ($entry->slug ?? ''));
            if ($slug === '') {
                continue;
            }

            $link = $this->build_link_data($entry);
            if ($link === null) {
                continue;
            }

            $links[$slug] = $link;
        }

        return $links;
    }

    /**
     * @return array<string, array<string, string>>
     */
    private function parse_product_nodes(SimpleXMLElement $parsed): array
    {
        $links = [];

        foreach ($parsed->product as $entry) {
            $link = $this->build_link_data($entry);
            if ($link === null) {
                continue;
            }

            $slugs = [];

            foreach ($entry->match_slugs->slug ?? [] as $slugEntry) {
                $slug = sanitize_title((string) $slugEntry);
                if ($slug !== '') {
                    $slugs[] = $slug;
                }
            }

            $singleSlug = sanitize_title((string) ($entry->slug ?? $entry->match_slug ?? ''));
            if ($singleSlug !== '') {
                $slugs[] = $singleSlug;
            }

            foreach (array_unique($slugs) as $slug) {
                $links[$slug] = $link;
            }
        }

        return $links;
    }

    /**
     * @return array<string, string>|null
     */
    private function build_link_data(SimpleXMLElement $entry): ?array
    {
        $defaultUrl = esc_url_raw((string) ($entry->url ?? ''));
        $signupUrl  = esc_url_raw((string) ($entry->signup_url ?? ''));
        $renewUrl   = esc_url_raw((string) ($entry->renew_url ?? $entry->renewal_url ?? ''));
        $resolved   = $renewUrl !== '' ? $renewUrl : ($signupUrl !== '' ? $signupUrl : $defaultUrl);

        if ($resolved === '') {
            return null;
        }

        return [
            'id'         => sanitize_text_field((string) ($entry->id ?? '')),
            'url'        => $resolved,
            'signup_url' => $signupUrl,
            'renew_url'  => $renewUrl,
        ];
    }

    /**
     * @param array<string, array<string, string>> $links
     */
    private function apply_affiliate_links(array $links): void
    {
        if (empty($links)) {
            return;
        }

        global $wpdb;
        $table = $wpdb->prefix . 'subscription_tracker';
        $rows  = $wpdb->get_results($wpdb->prepare("SELECT id, item_slug, item_name FROM {$table} LIMIT %d", 1000000), ARRAY_A);

        foreach ($rows as $row) {
            $slug = sanitize_title((string) ($row['item_slug'] ?? ''));

            if ($slug === '') {
                $slug = sanitize_title((string) ($row['item_name'] ?? ''));
            }

            if ($slug === '' || ! isset($links[$slug])) {
                continue;
            }

            $wpdb->update(
                $table,
                [
                    'affiliate_url' => esc_url_raw((string) ($links[$slug]['url'] ?? '')),
                    'affiliate_id'  => sanitize_text_field((string) ($links[$slug]['id'] ?? '')),
                    'updated_at'    => current_time('mysql'),
                ],
                ['id' => (int) $row['id']],
                ['%s', '%s', '%s'],
                ['%d']
            );
        }
    }
}
