<?php

if (! defined('ABSPATH')) {
    exit;
}

class WPST_Admin
{
    public function init(): void
    {
        add_action('admin_menu', [$this, 'register_menu']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue']);
        add_action('admin_post_wpst_save_subscription', [$this, 'save_subscription']);
        add_action('admin_post_wpst_delete_subscription', [$this, 'delete_subscription']);
        add_action('admin_post_wpst_scan', [$this, 'scan_subscriptions']);
        add_action('admin_post_wpst_save_settings', [$this, 'save_settings']);
        add_action('admin_post_wpst_export_csv', ['WPST_Export', 'download']);
        add_action('admin_post_wpst_export_ical', ['WPST_Export', 'download_ical']);
        add_action('admin_post_wpst_sync_pricing', [$this, 'sync_pricing']);
        add_action('admin_post_wpst_subscription_redirect', [$this, 'redirect_subscription']);
        add_action('wp_ajax_wpst_list_subscriptions', [$this, 'ajax_list_subscriptions']);
        add_action('wp_ajax_wpst_save_subscription', [$this, 'ajax_save_subscription']);
        add_action('wp_ajax_wpst_delete_subscription', [$this, 'ajax_delete_subscription']);
    }

    public function register_menu(): void
    {
        add_management_page(
            __('Subscription Tracker', 'wp-subscription-tracker'),
            __('Subscription Tracker', 'wp-subscription-tracker'),
            'manage_options',
            'wpst-subscription-tracker',
            [$this, 'render_page']
        );
    }

    public function enqueue(string $hook): void
    {
        if ($hook !== 'tools_page_wpst-subscription-tracker') {
            return;
        }

        wp_enqueue_style('wpst-admin', WPST_PLUGIN_URL . 'admin/css/wpst-admin.css', [], WPST_VERSION);
        wp_enqueue_script('wpst-admin', WPST_PLUGIN_URL . 'admin/js/wpst-admin.js', ['jquery'], WPST_VERSION, true);
        wp_localize_script('wpst-admin', 'wpstAdmin', [
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce'   => wp_create_nonce('wpst_ajax'),
            'i18n'    => [
                'reveal'             => __('Reveal', 'wp-subscription-tracker'),
                'hide'               => __('Hide', 'wp-subscription-tracker'),
                'nameRequired'       => __('Name is required.', 'wp-subscription-tracker'),
                'costInvalid'        => __('Cost must be 0 or greater.', 'wp-subscription-tracker'),
                'loadFailed'         => __('Failed to load subscriptions.', 'wp-subscription-tracker'),
                'saveFailed'         => __('Unable to save subscription.', 'wp-subscription-tracker'),
                'saveSuccess'        => __('Subscription saved.', 'wp-subscription-tracker'),
                'deleteConfirm'      => __('Delete this subscription?', 'wp-subscription-tracker'),
                'deleteFailed'       => __('Delete failed.', 'wp-subscription-tracker'),
                'deleteSuccess'      => __('Subscription deleted.', 'wp-subscription-tracker'),
                'editLoadFailed'     => __('Unable to load row for editing.', 'wp-subscription-tracker'),
                'emptyRows'          => __('No subscriptions found.', 'wp-subscription-tracker'),
                'resultsSummary'     => __('Showing %d subscriptions.', 'wp-subscription-tracker'),
                'resultsSummaryOne'  => __('Showing 1 subscription.', 'wp-subscription-tracker'),
                'resultsSummaryZero' => __('No subscriptions found for this filter.', 'wp-subscription-tracker'),
                'editModeOn'         => __('Edit mode enabled. Update fields and save to apply changes.', 'wp-subscription-tracker'),
            ],
        ]);
    }

    public function render_page(): void
    {
        if (! current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have permission to access this page.', 'wp-subscription-tracker'));
        }

        global $wpdb;
        $table = $wpdb->prefix . 'subscription_tracker';
        $rows  = $wpdb->get_results($wpdb->prepare("SELECT * FROM {$table} ORDER BY renewal_date ASC, item_name ASC LIMIT %d", 1000000), ARRAY_A);

        $totals = $this->calculate_totals($rows);
        $pricingSync = new WPST_Pricing_Sync();
        $priceAlerts = $pricingSync->get_price_alerts();

        include WPST_PLUGIN_DIR . 'templates/dashboard.php';
    }

    public function save_subscription(): void
    {
        $this->authorize_post('wpst_save_subscription');

        global $wpdb;
        $table = $wpdb->prefix . 'subscription_tracker';

        $data                = $this->build_subscription_data($_POST);
        $data['is_detected'] = 0;

        if ($data['item_name'] === '') {
            wp_safe_redirect(admin_url('tools.php?page=wpst-subscription-tracker&wpst_error=1'));
            exit;
        }

        $id = isset($_POST['id']) ? (int) $_POST['id'] : 0;

        if ($id > 0) {
            $wpdb->update($table, $data, ['id' => $id]);
        } else {
            $data['created_at'] = current_time('mysql');
            $wpdb->insert($table, $data);
        }

        wp_safe_redirect(admin_url('tools.php?page=wpst-subscription-tracker&wpst_saved=1'));
        exit;
    }

    public function delete_subscription(): void
    {
        $this->authorize_post('wpst_delete_subscription');

        global $wpdb;
        $table = $wpdb->prefix . 'subscription_tracker';
        $id    = isset($_POST['id']) ? (int) $_POST['id'] : 0;

        if ($id > 0) {
            $wpdb->delete($table, ['id' => $id]);
        }

        wp_safe_redirect(admin_url('tools.php?page=wpst-subscription-tracker&wpst_deleted=1'));
        exit;
    }

    public function scan_subscriptions(): void
    {
        $this->authorize_post('wpst_scan');

        $detector = new WPST_Detector();
        $items    = $detector->scan();

        global $wpdb;
        $table = $wpdb->prefix . 'subscription_tracker';

        foreach ($items as $item) {
            $exists = $wpdb->get_var($wpdb->prepare("SELECT id FROM {$table} WHERE item_slug = %s LIMIT 1", $item['item_slug']));
            if ($exists) {
                continue;
            }

            $item['status']      = isset($item['status']) ? $this->normalize_status((string) $item['status']) : 'active';
            $item['auto_renew']  = 1;
            $item['renewal_date'] = null;
            $item['license_key'] = '';
            $item['notes']       = '';
            $item['created_at']  = current_time('mysql');
            $item['updated_at']  = current_time('mysql');

            $item = $this->apply_affiliate_match($item);

            $wpdb->insert($table, $item);
        }

        wp_safe_redirect(admin_url('tools.php?page=wpst-subscription-tracker&wpst_scanned=1'));
        exit;
    }

    public function save_settings(): void
    {
        $this->authorize_post('wpst_save_settings');

        update_option('wpst_default_currency', sanitize_text_field((string) ($_POST['default_currency'] ?? 'CAD')));
        update_option('wpst_notification_email', sanitize_email((string) ($_POST['notification_email'] ?? get_option('admin_email'))));
        update_option('wpst_affiliate_disabled', isset($_POST['affiliate_disabled']) ? '1' : '0');
        update_option('wpst_pricing_api_base', esc_url_raw((string) ($_POST['pricing_api_base'] ?? 'https://shop.web321.co/v1/license_tracker/')));
        update_option('wpst_pricing_api_key', sanitize_text_field((string) ($_POST['pricing_api_key'] ?? '')));

        $siteToken = sanitize_text_field((string) ($_POST['site_token'] ?? ''));
        if ($siteToken === '') {
            $siteToken = wp_generate_password(32, false, false);
        }
        update_option('wpst_site_token', $siteToken);

        $intervals = array_map('sanitize_text_field', (array) ($_POST['reminder_intervals'] ?? []));
        update_option('wpst_reminder_intervals', $intervals);

        wp_safe_redirect(admin_url('tools.php?page=wpst-subscription-tracker&wpst_settings=1'));
        exit;
    }

    public function ajax_list_subscriptions(): void
    {
        $this->authorize_ajax();

        global $wpdb;
        $table = $wpdb->prefix . 'subscription_tracker';

        $filter   = sanitize_text_field((string) ($_POST['filter'] ?? 'all'));
        $search   = sanitize_text_field((string) ($_POST['search'] ?? ''));
        $sort     = sanitize_text_field((string) ($_POST['sort'] ?? 'renewal_asc'));
        $today    = gmdate('Y-m-d');
        $soonDate = gmdate('Y-m-d', strtotime('+30 days'));

        $where  = ['1=%d'];
        $params = [1];

        if ($filter === 'plugins') {
            $where[]  = 'item_type = %s';
            $params[] = 'plugin';
        } elseif ($filter === 'themes') {
            $where[]  = 'item_type = %s';
            $params[] = 'theme';
        } elseif ($filter === 'expiring') {
            $where[]  = 'renewal_date IS NOT NULL AND renewal_date BETWEEN %s AND %s';
            $params[] = $today;
            $params[] = $soonDate;
        } elseif ($filter === 'inactive') {
            $where[] = "status IN ('expired','cancelled')";
        } elseif ($filter === 'needs_review') {
            $where[]  = 'status = %s';
            $params[] = 'needs_review';
        }

        if ($search !== '') {
            $where[]  = '(item_name LIKE %s OR vendor LIKE %s OR item_slug LIKE %s)';
            $like     = '%' . $wpdb->esc_like($search) . '%';
            $params[] = $like;
            $params[] = $like;
            $params[] = $like;
        }

        $orderBy = 'renewal_date ASC, item_name ASC';
        if ($sort === 'name_asc') {
            $orderBy = 'item_name ASC';
        } elseif ($sort === 'name_desc') {
            $orderBy = 'item_name DESC';
        } elseif ($sort === 'cost_desc') {
            $orderBy = 'cost DESC, item_name ASC';
        } elseif ($sort === 'renewal_desc') {
            $orderBy = 'renewal_date DESC, item_name ASC';
        }

        $sql = "SELECT * FROM {$table} WHERE " . implode(' AND ', $where) . " ORDER BY {$orderBy}";

        $rows = $wpdb->get_results($wpdb->prepare($sql, ...$params), ARRAY_A);

        wp_send_json_success([
            'rows' => array_map([$this, 'normalize_row_for_json'], $rows),
        ]);
    }

    public function ajax_save_subscription(): void
    {
        $this->authorize_ajax();

        $data = $this->build_subscription_data($_POST);

        if ($data['item_name'] === '') {
            wp_send_json_error(['message' => __('Name is required.', 'wp-subscription-tracker')], 400);
        }

        global $wpdb;
        $table = $wpdb->prefix . 'subscription_tracker';
        $id    = isset($_POST['id']) ? (int) $_POST['id'] : 0;

        if ($id > 0) {
            $wpdb->update($table, $data, ['id' => $id]);
        } else {
            $data['created_at'] = current_time('mysql');
            $wpdb->insert($table, $data);
            $id = (int) $wpdb->insert_id;
        }

        $row = $wpdb->get_row($wpdb->prepare("SELECT * FROM {$table} WHERE id = %d", $id), ARRAY_A);
        if (! $row) {
            wp_send_json_error(['message' => __('Unable to save subscription.', 'wp-subscription-tracker')], 500);
        }

        wp_send_json_success([
            'row' => $this->normalize_row_for_json($row),
        ]);
    }

    public function ajax_delete_subscription(): void
    {
        $this->authorize_ajax();

        $id = isset($_POST['id']) ? (int) $_POST['id'] : 0;
        if ($id <= 0) {
            wp_send_json_error(['message' => __('Invalid subscription ID.', 'wp-subscription-tracker')], 400);
        }

        global $wpdb;
        $table = $wpdb->prefix . 'subscription_tracker';
        $wpdb->delete($table, ['id' => $id]);

        wp_send_json_success();
    }




    public function redirect_subscription(): void
    {
        if (! current_user_can('manage_options')) {
            wp_die(esc_html__('Insufficient permissions.', 'wp-subscription-tracker'));
        }

        $id      = isset($_GET['id']) ? (int) $_GET['id'] : 0;
        $context = sanitize_key((string) ($_GET['context'] ?? 'renew'));
        if (! in_array($context, ['renew', 'signup'], true)) {
            $context = 'renew';
        }

        check_admin_referer('wpst_subscription_redirect_' . $id . '_' . $context, 'wpst_nonce');

        if ($id <= 0) {
            wp_safe_redirect(admin_url('tools.php?page=wpst-subscription-tracker'));
            exit;
        }

        global $wpdb;
        $table = $wpdb->prefix . 'subscription_tracker';
        $row   = $wpdb->get_row($wpdb->prepare("SELECT id, item_slug, item_name, vendor_url, affiliate_url FROM {$table} WHERE id = %d", $id), ARRAY_A);

        if (! is_array($row)) {
            wp_safe_redirect(admin_url('tools.php?page=wpst-subscription-tracker'));
            exit;
        }

        $target = $this->resolve_redirect_target($row, $context);
        if ($target === '') {
            wp_safe_redirect(admin_url('tools.php?page=wpst-subscription-tracker'));
            exit;
        }

        wp_safe_redirect($target);
        exit;
    }
    public function sync_pricing(): void
    {
        if (! current_user_can('manage_options')) {
            wp_die(esc_html__('Insufficient permissions.', 'wp-subscription-tracker'));
        }

        check_admin_referer('wpst_sync_pricing');

        $pricingSync = new WPST_Pricing_Sync();
        $pricingSync->sync_all_pricing();

        wp_safe_redirect(admin_url('tools.php?page=wpst-subscription-tracker&wpst_pricing_synced=1'));
        exit;
    }

    /**
     * @param array<int, array<string, mixed>> $rows
     * @return array<string, mixed>
     */
    private function calculate_totals(array $rows): array
    {
        $monthly = 0.0;
        $annual  = 0.0;
        $active  = 0;
        $next    = null;

        foreach ($rows as $row) {
            $cost = (float) $row['cost'];
            if ($row['status'] === 'active') {
                $active++;
            }

            if ($row['billing_cycle'] === 'monthly') {
                $monthly += $cost;
                $annual += ($cost * 12);
            } elseif ($row['billing_cycle'] === 'annually') {
                $monthly += ($cost / 12);
                $annual += $cost;
            }

            if (! empty($row['renewal_date']) && ($next === null || $row['renewal_date'] < $next['renewal_date'])) {
                $next = $row;
            }
        }

        return [
            'monthly' => round($monthly, 2),
            'annual'  => round($annual, 2),
            'active'  => $active,
            'next'    => $next,
        ];
    }

    private function authorize_post(string $nonce_action): void
    {
        if (! current_user_can('manage_options')) {
            wp_die(esc_html__('Insufficient permissions.', 'wp-subscription-tracker'));
        }

        check_admin_referer($nonce_action);
    }

    private function authorize_ajax(): void
    {
        if (! current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Insufficient permissions.', 'wp-subscription-tracker')], 403);
        }

        check_ajax_referer('wpst_ajax', 'nonce');
    }

    /**
     * @param array<string, mixed> $input
     * @return array<string, mixed>
     */
    private function build_subscription_data(array $input): array
    {
        $data = [
            'item_type'     => $this->normalize_item_type((string) ($input['item_type'] ?? 'other')),
            'item_slug'     => sanitize_text_field((string) ($input['item_slug'] ?? '')),
            'item_name'     => sanitize_text_field((string) ($input['item_name'] ?? '')),
            'vendor'             => sanitize_text_field((string) ($input['vendor'] ?? '')),
            'vendor_url'         => esc_url_raw((string) ($input['vendor_url'] ?? '')),
            'handled_by'         => $this->normalize_handled_by((string) ($input['handled_by'] ?? 'vendor')),
            'management_company' => sanitize_text_field((string) ($input['management_company'] ?? '')),
            'cost'          => (float) ($input['cost'] ?? 0),
            'currency'      => sanitize_text_field((string) ($input['currency'] ?? 'CAD')),
            'billing_cycle' => $this->normalize_billing_cycle((string) ($input['billing_cycle'] ?? 'annually')),
            'renewal_date'  => $this->normalize_renewal_date((string) ($input['renewal_date'] ?? '')),
            'auto_renew'    => isset($input['auto_renew']) && (string) $input['auto_renew'] !== '0' ? 1 : 0,
            'license_key'   => WPST_Encryption::encrypt(sanitize_text_field((string) ($input['license_key'] ?? ''))),
            'notes'         => sanitize_textarea_field((string) ($input['notes'] ?? '')),
            'status'        => $this->normalize_status((string) ($input['status'] ?? 'active')),
            'updated_at'    => current_time('mysql'),
        ];

        if ($data['handled_by'] !== 'management_company') {
            $data['management_company'] = '';
        }

        if ($data['vendor_url'] === '') {
            $data['vendor_url'] = null;
        }

        return $this->apply_affiliate_match($data);
    }

    /**
     * @param array<string, mixed> $data
     * @return array<string, mixed>
     */
    private function apply_affiliate_match(array $data): array
    {
        if (get_option('wpst_affiliate_disabled', '0') === '1') {
            return $data;
        }

        $links = get_transient('wpst_affiliate_links');
        if (! is_array($links) || empty($links)) {
            return $data;
        }

        $slugCandidates = [
            sanitize_title((string) ($data['item_slug'] ?? '')),
            sanitize_title((string) ($data['item_name'] ?? '')),
        ];

        foreach ($slugCandidates as $slug) {
            if ($slug === '' || ! isset($links[$slug]) || ! is_array($links[$slug])) {
                continue;
            }

            $data['affiliate_url'] = esc_url_raw((string) ($links[$slug]['url'] ?? ''));
            $data['affiliate_id']  = sanitize_text_field((string) ($links[$slug]['id'] ?? ''));

            return $data;
        }

        return $data;
    }



    /**
     * @param array<string, mixed> $row
     */
    private function resolve_redirect_target(array $row, string $context): string
    {
        $vendorUrl = esc_url_raw((string) ($row['vendor_url'] ?? ''));

        if (get_option('wpst_affiliate_disabled', '0') === '1') {
            return $vendorUrl;
        }

        $affiliateUrl = esc_url_raw((string) ($row['affiliate_url'] ?? ''));
        $link         = $this->find_affiliate_link_for_row($row);
        $signupUrl    = esc_url_raw((string) ($link['signup_url'] ?? ''));
        $renewUrl     = esc_url_raw((string) ($link['renew_url'] ?? ''));

        if ($context === 'signup') {
            return $signupUrl !== '' ? $signupUrl : ($affiliateUrl !== '' ? $affiliateUrl : $vendorUrl);
        }

        if ($renewUrl !== '') {
            return $renewUrl;
        }

        return $affiliateUrl !== '' ? $affiliateUrl : $vendorUrl;
    }

    /**
     * @param array<string, mixed> $row
     * @return array<string, string>
     */
    private function find_affiliate_link_for_row(array $row): array
    {
        $links = get_transient('wpst_affiliate_links');
        if (! is_array($links)) {
            return [];
        }

        $candidates = [
            sanitize_title((string) ($row['item_slug'] ?? '')),
            sanitize_title((string) ($row['item_name'] ?? '')),
        ];

        foreach ($candidates as $slug) {
            if ($slug !== '' && isset($links[$slug]) && is_array($links[$slug])) {
                /** @var array<string, string> $link */
                $link = $links[$slug];
                return $link;
            }
        }

        return [];
    }
    private function normalize_handled_by(string $handledBy): string
    {
        $allowed   = ['vendor', 'management_company'];
        $handledBy = sanitize_text_field($handledBy);

        if (! in_array($handledBy, $allowed, true)) {
            return 'vendor';
        }

        return $handledBy;
    }

    private function normalize_status(string $status): string
    {
        $allowed = ['active', 'expired', 'cancelled', 'trial', 'needs_review'];
        $status  = sanitize_text_field($status);

        if (! in_array($status, $allowed, true)) {
            return 'active';
        }

        return $status;
    }

    private function normalize_item_type(string $itemType): string
    {
        $allowed  = ['plugin', 'theme', 'service', 'hosting', 'domain', 'other'];
        $itemType = sanitize_text_field($itemType);

        if (! in_array($itemType, $allowed, true)) {
            return 'other';
        }

        return $itemType;
    }

    private function normalize_billing_cycle(string $billingCycle): string
    {
        $allowed      = ['monthly', 'annually', 'lifetime', 'one-time'];
        $billingCycle = sanitize_text_field($billingCycle);

        if (! in_array($billingCycle, $allowed, true)) {
            return 'annually';
        }

        return $billingCycle;
    }

    private function normalize_renewal_date(string $renewalDate): ?string
    {
        $renewalDate = sanitize_text_field($renewalDate);
        if ($renewalDate === '') {
            return null;
        }

        $date = wp_date('Y-m-d', strtotime($renewalDate));
        if ($date === '1970-01-01') {
            return null;
        }

        return $date;
    }

    /**
     * @param array<string, mixed> $row
     * @return array<string, mixed>
     */
    private function normalize_row_for_json(array $row): array
    {
        $row['id']           = (int) $row['id'];
        $row['cost']         = (float) $row['cost'];
        $row['auto_renew']   = (int) $row['auto_renew'];
        $row['market_price'] = isset($row['market_price']) && $row['market_price'] !== null ? (float) $row['market_price'] : null;
        $row['license_key']  = WPST_Encryption::decrypt((string) ($row['license_key'] ?? ''));
        $row['affiliate_on'] = get_option('wpst_affiliate_disabled', '0') !== '1';

        return $row;
    }
}
