<?php

declare(strict_types=1);

namespace WPST\PricingApi;

use PDO;

final class ProductRepository
{
    private PDO $pdo;

    public function __construct(PDO $pdo)
    {
        $this->pdo = $pdo;
    }

    /** @return array<string, mixed> */
    public function listProducts(array $filters): array
    {
        $page = max(1, (int) ($filters['page'] ?? 1));
        $limit = min(250, max(1, (int) ($filters['limit'] ?? 100)));
        $offset = ($page - 1) * $limit;

        $where = ['p.active = 1'];
        $params = [];

        if (! empty($filters['type'])) {
            $where[] = 'p.type = :type';
            $params[':type'] = (string) $filters['type'];
        }
        if (! empty($filters['marketplace'])) {
            $where[] = 'p.marketplace = :marketplace';
            $params[':marketplace'] = (string) $filters['marketplace'];
        }

        $whereSql = implode(' AND ', $where);

        $countStmt = $this->pdo->prepare("SELECT COUNT(*) FROM products p WHERE {$whereSql}");
        $countStmt->execute($params);
        $total = (int) $countStmt->fetchColumn();

        $sql = "SELECT p.id as product_id, p.slug, p.name, p.affiliate_url, p.last_updated,
                MAX(CASE WHEN pp.billing_cycle = 'annual' THEN pp.price END) AS annual,
                MAX(CASE WHEN pp.billing_cycle = 'monthly' THEN pp.price END) AS monthly,
                MAX(CASE WHEN pp.billing_cycle = 'lifetime' THEN pp.price END) AS lifetime,
                MAX(pp.currency) AS currency
            FROM products p
            LEFT JOIN product_prices pp ON pp.product_id = p.id AND pp.is_current = 1
            WHERE {$whereSql}
            GROUP BY p.id, p.slug, p.name, p.affiliate_url, p.last_updated
            ORDER BY p.last_updated DESC
            LIMIT :limit OFFSET :offset";

        $stmt = $this->pdo->prepare($sql);
        foreach ($params as $key => $value) {
            $stmt->bindValue($key, $value);
        }
        $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
        $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
        $stmt->execute();

        return [
            'data' => array_map([$this, 'mapProductRow'], $stmt->fetchAll()),
            'meta' => [
                'total' => $total,
                'page' => $page,
                'limit' => $limit,
                'pages' => (int) ceil($total / $limit),
            ],
        ];
    }

    /** @return array<string, mixed>|null */
    public function getProductBySlug(string $slug): ?array
    {
        $stmt = $this->pdo->prepare(
            "SELECT p.id as product_id, p.slug, p.name, p.provider, p.marketplace, p.type, p.affiliate_url, p.product_url, p.last_updated,
                pp.billing_cycle, pp.price, pp.currency
            FROM products p
            LEFT JOIN product_prices pp ON pp.product_id = p.id AND pp.is_current = 1
            WHERE p.slug = :slug AND p.active = 1"
        );
        $stmt->execute([':slug' => $slug]);
        $rows = $stmt->fetchAll();
        if (! $rows) {
            return null;
        }

        $base = $rows[0];
        $pricing = [];
        foreach ($rows as $row) {
            if (! empty($row['billing_cycle'])) {
                $pricing[(string) $row['billing_cycle']] = (float) $row['price'];
                $pricing['currency'] = (string) $row['currency'];
            }
        }

        return [
            'product_id' => $base['product_id'],
            'slug' => $base['slug'],
            'name' => $base['name'],
            'provider' => $base['provider'],
            'marketplace' => $base['marketplace'],
            'type' => $base['type'],
            'pricing' => $pricing,
            'url' => $base['product_url'],
            'affiliate_url' => $base['affiliate_url'],
            'last_updated' => $base['last_updated'],
        ];
    }

    /** @return array<int, array<string, mixed>> */
    public function getBatchBySlugs(array $slugs): array
    {
        if (empty($slugs)) {
            return [];
        }

        $normalized = array_values(array_unique(array_filter(array_map(static function ($slug): string {
            return strtolower(trim((string) $slug));
        }, $slugs))));
        if (empty($normalized)) {
            return [];
        }

        $in = implode(',', array_fill(0, count($normalized), '?'));
        $stmt = $this->pdo->prepare(
            "SELECT p.slug FROM products p WHERE p.slug IN ({$in}) AND p.active = 1"
        );
        $stmt->execute($normalized);
        $foundSlugs = array_map(static fn ($row): string => (string) $row['slug'], $stmt->fetchAll());

        $data = [];
        foreach ($foundSlugs as $slug) {
            $product = $this->getProductBySlug($slug);
            if ($product !== null) {
                $data[] = $product;
            }
        }

        return $data;
    }

    /** @return array<string, mixed> */
    public function search(string $query, array $filters): array
    {
        $q = '%' . strtolower($query) . '%';
        $type = (string) ($filters['type'] ?? '');

        $sql = "SELECT p.slug FROM products p WHERE p.active = 1 AND (LOWER(p.slug) LIKE :q OR LOWER(p.name) LIKE :q)";
        $params = [':q' => $q];
        if ($type !== '') {
            $sql .= ' AND p.type = :type';
            $params[':type'] = $type;
        }
        $sql .= ' ORDER BY p.last_updated DESC LIMIT 50';

        $stmt = $this->pdo->prepare($sql);
        $stmt->execute($params);

        $out = [];
        foreach ($stmt->fetchAll() as $row) {
            $product = $this->getProductBySlug((string) $row['slug']);
            if ($product !== null) {
                $out[] = $product;
            }
        }

        return ['data' => $out];
    }

    /** @return array<string, mixed> */
    public function changes(?string $since, bool $includeMinor): array
    {
        $sql = "SELECT pc.product_id, p.slug, p.name, pc.change_type, pc.field, pc.old_value, pc.new_value, pc.percent_change, pc.detected_at
            FROM price_changes pc
            INNER JOIN products p ON p.id = pc.product_id
            WHERE p.active = 1";
        $params = [];
        if ($since) {
            $sql .= ' AND pc.detected_at >= :since';
            $params[':since'] = $since;
        }
        if (! $includeMinor) {
            $sql .= ' AND ABS(pc.percent_change) >= 1';
        }
        $sql .= ' ORDER BY pc.detected_at DESC LIMIT 200';

        $stmt = $this->pdo->prepare($sql);
        $stmt->execute($params);
        $changes = $stmt->fetchAll();

        $summary = [
            'total_changes' => count($changes),
            'price_increases' => 0,
            'price_decreases' => 0,
            'new_products' => 0,
        ];
        foreach ($changes as $change) {
            if (($change['change_type'] ?? '') === 'increase') {
                $summary['price_increases']++;
            } elseif (($change['change_type'] ?? '') === 'decrease') {
                $summary['price_decreases']++;
            } elseif (($change['change_type'] ?? '') === 'new') {
                $summary['new_products']++;
            }
        }

        return ['changes' => $changes, 'summary' => $summary];
    }

    /** @param array<string, mixed> $row */
    private function mapProductRow(array $row): array
    {
        return [
            'product_id' => $row['product_id'],
            'slug' => $row['slug'],
            'name' => $row['name'],
            'pricing' => [
                'monthly' => $row['monthly'] !== null ? (float) $row['monthly'] : null,
                'annual' => $row['annual'] !== null ? (float) $row['annual'] : null,
                'lifetime' => $row['lifetime'] !== null ? (float) $row['lifetime'] : null,
                'currency' => $row['currency'] ?: 'USD',
            ],
            'affiliate_url' => $row['affiliate_url'],
            'last_updated' => $row['last_updated'],
        ];
    }
}
