(function ($) {
    'use strict';

    var subscriptions = wpstInventory.subscriptions || {};
    var pluginMeta = wpstInventory.pluginMeta || {};

    function esc(value) {
        return $('<div>').text(value || '').html();
    }

    function t(key, fallback) {
        if (wpstInventory.i18n && wpstInventory.i18n[key]) {
            return wpstInventory.i18n[key];
        }

        return fallback;
    }

    function isPromo(notes) {
        return /(sale|promo|promotion|discount|deal)/i.test(String(notes || ''));
    }

    function daysUntil(dateString) {
        if (!dateString) {
            return null;
        }

        var now = new Date();
        var target = new Date(dateString + 'T00:00:00');
        if (Number.isNaN(target.getTime())) {
            return null;
        }

        var oneDay = 24 * 60 * 60 * 1000;
        return Math.ceil((target - new Date(now.getFullYear(), now.getMonth(), now.getDate())) / oneDay);
    }

    function parseVendorMeta($container) {
        var data = { vendorName: '', vendorUrl: '' };
        var $authorLink = $container.find('.plugin-author a').first();
        if ($authorLink.length) {
            data.vendorName = $.trim($authorLink.text());
            data.vendorUrl = String($authorLink.attr('href') || '');
            return data;
        }

        var text = $.trim($container.find('.plugin-author').first().text()).replace(/^By\s+/i, '');
        data.vendorName = text;
        return data;
    }

    function pluginMetadataFor(slug) {
        var data = pluginMeta[slug] || {};

        return {
            vendorName: $.trim(String(data.vendor_name || '')),
            vendorUrl: $.trim(String(data.vendor_url || ''))
        };
    }

    function buildDefaultContext(itemType, itemSlug, itemName, vendorName, vendorUrl) {
        return {
            item_type: itemType,
            item_slug: itemSlug,
            item_name: itemName,
            vendor_name: vendorName || '',
            vendor_url: vendorUrl || ''
        };
    }

    function modalHtml() {
        return '<div id="wpst-inventory-modal" class="wpst-modal" style="display:none;">' +
            '<div class="wpst-modal-backdrop"></div>' +
            '<div class="wpst-modal-dialog" role="dialog" aria-modal="true" aria-labelledby="wpst-modal-title">' +
            '<h2 id="wpst-modal-title"></h2>' +
            '<div id="wpst-modal-notice" class="notice inline" style="display:none;"></div>' +
            '<form id="wpst-inventory-form">' +
            '<input type="hidden" name="id" value="" />' +
            '<input type="hidden" name="item_type" value="" />' +
            '<input type="hidden" name="item_slug" value="" />' +
            '<div class="wpst-form-row"><label for="wpst-item-name">Name</label><input id="wpst-item-name" type="text" name="item_name" required /></div>' +
            '<div class="wpst-form-row"><label for="wpst-handled-by">Track As</label><select id="wpst-handled-by" name="handled_by"><option value="vendor">Vendor</option><option value="management_company">Website Management Company</option></select></div>' +
            '<div class="wpst-form-row"><label for="wpst-vendor">Vendor Name</label><input id="wpst-vendor" type="text" name="vendor" /></div>' +
            '<div class="wpst-form-row"><label for="wpst-vendor-url">Vendor URL</label><div><input id="wpst-vendor-url" type="url" name="vendor_url" placeholder="https://example.com" /><a id="wpst-vendor-url-preview" href="#" target="_blank" rel="noopener noreferrer" style="display:none; margin-top:6px;">Open vendor website</a></div></div>' +
            '<div class="wpst-form-row"><label for="wpst-management-company">Management Company</label><input id="wpst-management-company" type="text" name="management_company" placeholder="Web321" /></div>' +
            '<div class="wpst-form-row"><label for="wpst-cost">Cost</label><input id="wpst-cost" type="number" name="cost" min="0" step="0.01" /></div>' +
            '<div class="wpst-form-row"><label for="wpst-currency">Currency</label><select id="wpst-currency" name="currency"></select></div>' +
            '<div class="wpst-form-row"><label for="wpst-billing-cycle">Billing Cycle</label><select id="wpst-billing-cycle" name="billing_cycle"><option value="monthly">Monthly</option><option value="annually">Annually</option><option value="lifetime">Lifetime</option><option value="one-time">One-time</option></select></div>' +
            '<div class="wpst-form-row"><label for="wpst-renewal-date">Renewal Date</label><input id="wpst-renewal-date" type="date" name="renewal_date" /></div>' +
            '<div class="wpst-form-row"><label for="wpst-status">Status</label><select id="wpst-status" name="status"><option value="active">Active</option><option value="trial">Trial</option><option value="needs_review">Needs Review</option><option value="expired">Expired</option><option value="cancelled">Cancelled</option></select></div>' +
            '<div class="wpst-form-row"><label for="wpst-notes">Notes</label><textarea id="wpst-notes" name="notes" rows="3"></textarea></div>' +
            '<p class="wpst-modal-actions"><button type="submit" class="button button-primary" id="wpst-modal-save"></button> <button type="button" class="button" id="wpst-modal-cancel"></button></p>' +
            '</form></div></div>';
    }

    function statusHtml(ctx, row) {
        if (!row) {
            return '<span class="wpst-status-muted">' + esc(t('noAction', 'No urgent updates.')) + '</span>';
        }

        var bits = [];
        var d = daysUntil(row.renewal_date);
        if (d !== null && d >= 0 && d <= 30) {
            bits.push('<a href="#" class="wpst-track-trigger" data-item-type="' + esc(ctx.item_type) + '" data-item-slug="' + esc(ctx.item_slug) + '" data-item-name="' + esc(ctx.item_name) + '">' + esc(t('expiringSoon', 'Expires in %d day(s): %s').replace('%d', String(d)).replace('%s', row.renewal_date)) + '</a>');
        }

        if (isPromo(row.notes)) {
            bits.push('<a href="#" class="wpst-track-trigger" data-item-type="' + esc(ctx.item_type) + '" data-item-slug="' + esc(ctx.item_slug) + '" data-item-name="' + esc(ctx.item_name) + '">' + esc(t('promotionAvailable', 'Promotion noted. Review offer details.')) + '</a>');
        }

        if (!bits.length) {
            return '<span class="wpst-status-muted">' + esc(t('noAction', 'No urgent updates.')) + '</span>';
        }

        return bits.join(' · ');
    }

    function subscriptionFor(type, slug) {
        return subscriptions[type + ':' + slug] || null;
    }

    function createTrackLink(ctx) {
        return '<a href="#" class="wpst-track-trigger" data-item-type="' + esc(ctx.item_type) + '" data-item-slug="' + esc(ctx.item_slug) + '" data-item-name="' + esc(ctx.item_name) + '" data-vendor-name="' + esc(ctx.vendor_name) + '" data-vendor-url="' + esc(ctx.vendor_url) + '">' + esc(t('track', 'Track')) + '</a>';
    }

    function addPluginEnhancements() {
        $('#the-list tr[data-plugin]').each(function () {
            var $row = $(this);
            var pluginFile = String($row.data('plugin') || '');
            var slug = pluginFile.split('/')[0].replace('.php', '');
            var name = $.trim($row.find('.plugin-title strong').first().text());
            if (!slug || !name) {
                return;
            }

            var vendorMeta = pluginMetadataFor(slug);
            if (!vendorMeta.vendorName && !vendorMeta.vendorUrl) {
                vendorMeta = parseVendorMeta($row);
            }
            var ctx = buildDefaultContext('plugin', slug, name, vendorMeta.vendorName, vendorMeta.vendorUrl);
            var row = subscriptionFor('plugin', slug);

            var $actions = $row.find('.row-actions').first();
            if ($actions.length && !$actions.find('.wpst-track-trigger').length) {
                $actions.append(' | ' + createTrackLink(ctx));
            }

            var pluginStatus = '<div class="wpst-quick-status">' + statusHtml(ctx, row) + '</div>';
            if (!$row.find('.wpst-quick-status').length) {
                $row.find('.plugin-title').append(pluginStatus);
            } else {
                $row.find('.wpst-quick-status').replaceWith(pluginStatus);
            }
        });
    }

    function addThemeEnhancements() {
        $('.theme').each(function () {
            var $theme = $(this);
            var slug = String($theme.data('slug') || '');
            var name = $.trim($theme.find('.theme-name').first().text());
            if (!slug || !name) {
                return;
            }

            var ctx = buildDefaultContext('theme', slug, name, '', '');
            var row = subscriptionFor('theme', slug);

            if (!$theme.find('.wpst-theme-tools').length) {
                $theme.find('.theme-id-container').append('<p class="wpst-theme-tools">' + createTrackLink(ctx) + '</p>');
            }

            var themeStatus = '<div class="wpst-quick-status">' + statusHtml(ctx, row) + '</div>';
            if (!$theme.find('.wpst-quick-status').length) {
                $theme.find('.theme-id-container').append(themeStatus);
            } else {
                $theme.find('.wpst-quick-status').replaceWith(themeStatus);
            }
        });
    }


    function buildCurrencyOptions(selectedCurrency) {
        var currencies = (wpstInventory.defaults && wpstInventory.defaults.currencies) || ['CAD', 'USD', 'EUR', 'GBP', 'AUD'];
        var selected = String(selectedCurrency || wpstInventory.defaults.currency || 'CAD').toUpperCase();
        var seen = {};

        return currencies.reduce(function (html, currency) {
            var code = String(currency || '').toUpperCase();
            if (!code || seen[code]) {
                return html;
            }

            seen[code] = true;
            return html + '<option value="' + esc(code) + '"' + (code === selected ? ' selected="selected"' : '') + '>' + esc(code) + '</option>';
        }, '');
    }


    function refreshVendorPreview() {
        var url = $.trim(String($('#wpst-vendor-url').val() || ''));
        var $link = $('#wpst-vendor-url-preview');
        if (!url) {
            $link.hide().attr('href', '#');
            return;
        }

        $link.attr('href', url).show();
    }

    function toggleHandledByFields() {
        var handledBy = $('#wpst-handled-by').val();
        var isManaged = handledBy === 'management_company';
        $('#wpst-management-company').prop('disabled', !isManaged).closest('.wpst-form-row').toggle(isManaged);
        $('#wpst-vendor, #wpst-vendor-url').prop('disabled', isManaged);
    }

    function fillForm(ctx, row) {
        var values = row || {};
        var $form = $('#wpst-inventory-form');

        $form.find('[name="id"]').val(values.id || '');
        $form.find('[name="item_type"]').val(ctx.item_type);
        $form.find('[name="item_slug"]').val(ctx.item_slug);
        $form.find('[name="item_name"]').val(values.item_name || ctx.item_name || '');
        $form.find('[name="handled_by"]').val(values.handled_by || 'vendor');
        $form.find('[name="vendor"]').val(values.vendor || ctx.vendor_name || '');
        $form.find('[name="vendor_url"]').val(values.vendor_url || ctx.vendor_url || '');
        $form.find('[name="management_company"]').val(values.management_company || '');
        $form.find('[name="cost"]').val(values.cost || '0.00');
        $form.find('[name="currency"]').html(buildCurrencyOptions(values.currency));
        $form.find('[name="billing_cycle"]').val(values.billing_cycle || 'annually');
        $form.find('[name="renewal_date"]').val(values.renewal_date || '');
        $form.find('[name="status"]').val(values.status || 'active');
        $form.find('[name="notes"]').val(values.notes || '');

        toggleHandledByFields();
        refreshVendorPreview();

        $('#wpst-modal-title').text(values.id ? t('modalTitleEdit', 'Update Subscription') : t('modalTitleNew', 'Track Subscription'));
        $('#wpst-modal-save').text(t('save', 'Save Subscription Info'));
        $('#wpst-modal-cancel').text(t('cancel', 'Cancel'));
    }

    function showModal(ctx) {
        var row = subscriptionFor(ctx.item_type, ctx.item_slug);
        fillForm(ctx, row);
        $('#wpst-modal-notice').hide();
        $('#wpst-inventory-modal').show();
    }

    function hideModal() {
        $('#wpst-inventory-modal').hide();
    }

    function modalNotice(message, isError) {
        $('#wpst-modal-notice').removeClass('notice-success notice-error').addClass(isError ? 'notice-error' : 'notice-success').html('<p>' + esc(message) + '</p>').show();
    }

    function serializeForm() {
        var payload = {
            action: 'wpst_save_subscription',
            nonce: wpstInventory.nonce
        };

        $('#wpst-inventory-form').serializeArray().forEach(function (item) {
            payload[item.name] = item.value;
        });

        payload.auto_renew = '1';
        return payload;
    }

    $(function () {
        $('body').append(modalHtml());

        if (wpstInventory.screen === 'plugins.php') {
            addPluginEnhancements();
        }

        if (wpstInventory.screen === 'themes.php') {
            addThemeEnhancements();
            var observer = new MutationObserver(addThemeEnhancements);
            observer.observe(document.body, { childList: true, subtree: true });
        }

        $(document).on('click', '.wpst-track-trigger', function (event) {
            event.preventDefault();
            showModal({
                item_type: $(this).data('item-type'),
                item_slug: $(this).data('item-slug'),
                item_name: $(this).data('item-name'),
                vendor_name: $(this).data('vendor-name'),
                vendor_url: $(this).data('vendor-url')
            });
        });

        $(document).on('change', '#wpst-handled-by', toggleHandledByFields);
        $(document).on('input', '#wpst-vendor-url', refreshVendorPreview);

        $(document).on('click', '#wpst-modal-cancel, .wpst-modal-backdrop', hideModal);

        $(document).on('keydown', function (event) {
            if (event.key === 'Escape' && $('#wpst-inventory-modal').is(':visible')) {
                hideModal();
            }
        });

        $('#wpst-inventory-form').on('submit', function (event) {
            event.preventDefault();
            var $save = $('#wpst-modal-save');
            $save.prop('disabled', true).text(t('saving', 'Saving…'));

            $.post(wpstInventory.ajaxUrl, serializeForm()).done(function (response) {
                if (!response.success || !response.data || !response.data.row) {
                    modalNotice(t('saveFailed', 'Unable to save subscription.'), true);
                    return;
                }

                var row = response.data.row;
                subscriptions[row.item_type + ':' + row.item_slug] = row;
                modalNotice(t('saveSuccess', 'Subscription saved.'), false);
                if (wpstInventory.screen === 'plugins.php') {
                    addPluginEnhancements();
                } else {
                    addThemeEnhancements();
                }
                window.setTimeout(hideModal, 500);
            }).fail(function () {
                modalNotice(t('saveFailed', 'Unable to save subscription.'), true);
            }).always(function () {
                $save.prop('disabled', false).text(t('save', 'Save Subscription Info'));
            });
        });
    });
})(jQuery);
