(function ($) {
    'use strict';

    function escapeHtml(value) {
        return $('<div>').text(value || '').html();
    }

    function i18n(key, fallback) {
        if (wpstAdmin && wpstAdmin.i18n && wpstAdmin.i18n[key]) {
            return wpstAdmin.i18n[key];
        }

        return fallback;
    }

    function updateResultsSummary(count) {
        var text;

        if (count <= 0) {
            text = i18n('resultsSummaryZero', 'No subscriptions found for this filter.');
        } else if (count === 1) {
            text = i18n('resultsSummaryOne', 'Showing 1 subscription.');
        } else {
            text = i18n('resultsSummary', 'Showing %d subscriptions.').replace('%d', String(count));
        }

        $('#wpst-results-summary').text(text);
    }

    function rowActions(row) {
        var links = [];
        if (row.affiliate_on && row.affiliate_url) {
            links.push('<a href="' + escapeHtml(row.affiliate_url) + '" target="_blank" rel="nofollow sponsored noopener">Renew via Web321</a>');
        }

        links.push('<button type="button" class="button-link wpst-edit-row">Edit</button>');
        links.push('<button type="button" class="button-link-delete wpst-delete-row" data-id="' + parseInt(row.id, 10) + '">Delete</button>');

        return links.join(' | ');
    }

    function buildRow(row) {
        var renewal = row.renewal_date ? row.renewal_date : '-';
        var encoded = encodeURIComponent(JSON.stringify(row));

        var market = row.market_price !== null && row.market_price !== undefined && row.market_price !== '' ? Number(row.market_price).toFixed(2) + ' ' + escapeHtml(row.currency) : '—';

        return '<tr data-row="' + encoded + '">' +
            '<td>' + escapeHtml(row.item_name) + '</td>' +
            '<td>' + escapeHtml(row.vendor) + '</td>' +
            '<td>' + Number(row.cost).toFixed(2) + ' ' + escapeHtml(row.currency) + '</td>' +
            '<td>' + market + '</td>' +
            '<td>' + escapeHtml(row.billing_cycle) + '</td>' +
            '<td>' + escapeHtml(renewal) + '</td>' +
            '<td>' + escapeHtml(row.status) + '</td>' +
            '<td>' + rowActions(row) + '</td>' +
            '</tr>';
    }

    function showNotice(message, isError) {
        var $notice = $('#wpst-inline-notice');
        $notice.removeClass('notice-success notice-error').addClass(isError ? 'notice-error' : 'notice-success');
        $notice.html('<p>' + escapeHtml(message) + '</p>').show();
    }

    function validateForm() {
        var name = $('#wpst-subscription-form [name="item_name"]').val();
        var cost = parseFloat($('#wpst-subscription-form [name="cost"]').val());

        if (!name || !String(name).trim()) {
            showNotice(i18n('nameRequired', 'Name is required.'), true);
            return false;
        }

        if (Number.isNaN(cost) || cost < 0) {
            showNotice(i18n('costInvalid', 'Cost must be 0 or greater.'), true);
            return false;
        }

        return true;
    }

    function formPayload() {
        var data = {
            action: 'wpst_save_subscription',
            nonce: wpstAdmin.nonce
        };

        $('#wpst-subscription-form').serializeArray().forEach(function (item) {
            data[item.name] = item.value;
        });

        data.auto_renew = $('#wpst-subscription-form [name="auto_renew"]').is(':checked') ? '1' : '0';

        return data;
    }

    function resetForm() {
        var form = document.getElementById('wpst-subscription-form');
        if (form) {
            form.reset();
            $('#wpst-subscription-form [name="id"]').val('');
            $('#wpst-cancel-edit').hide();
            $('#wpst-subscription-form [name="license_key"]').attr('type', 'password');
            $('#wpst-toggle-license').attr('aria-pressed', 'false').text(i18n('reveal', 'Reveal'));
        }
    }

    function loadRows() {
        $.post(wpstAdmin.ajaxUrl, {
            action: 'wpst_list_subscriptions',
            nonce: wpstAdmin.nonce,
            filter: $('.wpst-filter-buttons .is-active').data('filter') || 'all',
            sort: $('#wpst-sort').val(),
            search: $('#wpst-search').val()
        }).done(function (response) {
            if (!response.success) {
                showNotice(response.data && response.data.message ? response.data.message : i18n('loadFailed', 'Failed to load subscriptions.'), true);
                return;
            }

            var rows = response.data.rows || [];
            var html = rows.length ? rows.map(buildRow).join('') : '<tr><td colspan="8">' + escapeHtml(i18n('emptyRows', 'No subscriptions found.')) + '</td></tr>';
            updateResultsSummary(rows.length);
            $('#wpst-subscription-tbody').html(html);
        }).fail(function () {
            showNotice(i18n('loadFailed', 'Failed to load subscriptions.'), true);
        });
    }

    $(function () {
        $('#wpst-subscription-form').on('submit', function (event) {
            event.preventDefault();

            if (!validateForm()) {
                return;
            }

            $.post(wpstAdmin.ajaxUrl, formPayload()).done(function (response) {
                if (!response.success) {
                    showNotice(response.data && response.data.message ? response.data.message : i18n('saveFailed', 'Unable to save subscription.'), true);
                    return;
                }

                showNotice(i18n('saveSuccess', 'Subscription saved.'), false);
                resetForm();
                loadRows();
            }).fail(function () {
                showNotice(i18n('saveFailed', 'Unable to save subscription.'), true);
            });
        });

        $('#wpst-subscription-tbody').on('click', '.wpst-delete-row', function () {
            var id = $(this).data('id');
            if (!window.confirm(i18n('deleteConfirm', 'Delete this subscription?'))) {
                return;
            }

            $.post(wpstAdmin.ajaxUrl, {
                action: 'wpst_delete_subscription',
                nonce: wpstAdmin.nonce,
                id: id
            }).done(function (response) {
                if (!response.success) {
                    showNotice(response.data && response.data.message ? response.data.message : i18n('deleteFailed', 'Delete failed.'), true);
                    return;
                }

                showNotice(i18n('deleteSuccess', 'Subscription deleted.'), false);
                loadRows();
            }).fail(function () {
                showNotice(i18n('deleteFailed', 'Delete failed.'), true);
            });
        });

        $('#wpst-subscription-tbody').on('click', '.wpst-edit-row', function () {
            var encoded = $(this).closest('tr').attr('data-row') || '';
            var row;

            try {
                row = JSON.parse(decodeURIComponent(encoded));
            } catch (error) {
                showNotice(i18n('editLoadFailed', 'Unable to load row for editing.'), true);
                return;
            }

            var $form = $('#wpst-subscription-form');
            Object.keys(row).forEach(function (key) {
                var $field = $form.find('[name="' + key + '"]');
                if ($field.length) {
                    $field.val(row[key]);
                }
            });

            $form.find('[name="auto_renew"]').prop('checked', Number(row.auto_renew) === 1);
            $('#wpst-cancel-edit').show();
            window.scrollTo({ top: $form.offset().top - 80, behavior: 'smooth' });
            showNotice(i18n('editModeOn', 'Edit mode enabled. Update fields and save to apply changes.'), false);
        });

        $('#wpst-cancel-edit').on('click', resetForm);

        $('#wpst-toggle-license').on('click', function () {
            var $input = $('#wpst-subscription-form [name="license_key"]');
            var reveal = $input.attr('type') === 'password';
            $input.attr('type', reveal ? 'text' : 'password');
            $(this).attr('aria-pressed', reveal ? 'true' : 'false').text(reveal ? i18n('hide', 'Hide') : i18n('reveal', 'Reveal'));
        });

        $('.wpst-filter-buttons button').on('click', function () {
            $('.wpst-filter-buttons button').removeClass('is-active');
            $(this).addClass('is-active');
            loadRows();
        });

        $('#wpst-sort').on('change', loadRows);

        var searchTimer;
        $('#wpst-search').on('input', function () {
            window.clearTimeout(searchTimer);
            searchTimer = window.setTimeout(loadRows, 250);
        });
    });
})(jQuery);
