<?php
/*
Plugin Name: Share a Draft Improved
Description: Share private preview links to your drafts with others. Improved from code from the 2010s.
Plugin URI: https://web321.co/plugins/
Donate: https://www.paypal.com/paypalme/web321co/20/
Version: 1.0.1
Author: dewolfe001
Author URI: https://web321.co/
Tags: post, post, draft, posts, drafts, share, sharing, publishing
Requires at least: 5.0
Tested up to: 6.7
Requires PHP: 7.4
License: GPLv2 or later
License URI: http://www.gnu.org/licenses/gpl-2.0.html
Text Domain: w321_sharedraft
Domain Path: /languages
*/

defined('ABSPATH') || exit;

define('WSD_BLOCK_NAME', plugin_basename(__FILE__));
define('WSD_BLOCK_DIR', plugin_dir_path(__FILE__));
define('WSD_BLOCK_URL', plugin_dir_url(__FILE__));
define('WSD_BLOCK_VERSION', '1.0');

add_filter('plugin_row_meta', 'wsd_block_row_meta', 10, 2);

/**
 * Add a donation link to plugin row meta.
 */
function wsd_block_row_meta($links, $file) {
    if (WSD_BLOCK_NAME === $file) {
        $row_meta = [
            'donate' => '<a href="' . esc_url('https://www.paypal.com/paypalme/web321co/20/') . '" target="_blank" aria-label="' . esc_attr__('Donate', 'wsd-block') . '">' . esc_html__('Donate', 'wsd-block') . '</a>'
        ];
        return array_merge($links, $row_meta);
    }
    return (array) $links;
}

if (!class_exists('Share_a_Draft')) :

class Share_a_Draft {
    private $admin_options_name = 'ShareADraft_options';
    private $admin_options = [];
    private $user_options = [];
    private $shared_post = null;

    public function __construct() {
        add_action('init', [$this, 'init']);
    }

    public function init() {
        // Add necessary hooks
        add_action('admin_menu', [$this, 'add_admin_pages']);
        add_filter('pre_get_posts', [$this, 'allow_draft_preview']);
        add_filter('the_posts', [$this, 'filter_shared_posts'], 10, 2);

        // Load admin options
        $this->admin_options = $this->get_admin_options();
        $this->admin_options = $this->clear_expired($this->admin_options);
        $this->save_admin_options();

        // Load translations
        load_plugin_textdomain('w321_sharedraft', false, dirname(plugin_basename(__FILE__)) . '/languages');

        // Admin page specific initializations
        if (is_admin() && isset($_GET['page']) && sanitize_text_field($_GET['page']) === plugin_basename(__FILE__)) {
            add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_scripts']);
        }
    }

    public function enqueue_admin_scripts() {
        wp_enqueue_script('jquery');
        wp_add_inline_script('jquery', $this->get_admin_javascript());
        wp_add_inline_style('admin', $this->get_admin_css());
    }

    private function get_admin_javascript() {
        return '
            (function ($) {
                $(function () {
                    $("form.shareadraft-extend").hide();
                    $("a.shareadraft-extend").show();
                    $("a.shareadraft-extend-cancel").show().css("display", "inline");
                });
                window.shareadraft = {
                    toggle_extend: function (key) {
                        $("#shareadraft-extend-form-" + key).show();
                        $("#shareadraft-extend-link-" + key).hide();
                        $("#shareadraft-extend-form-" + key + " input[name=\"expires\"]").focus();
                    },
                    cancel_extend: function (key) {
                        $("#shareadraft-extend-form-" + key).hide();
                        $("#shareadraft-extend-link-" + key).show();
                    }
                };
            })(jQuery);
        ';
    }

    private function get_admin_css() {
        return '
            a.shareadraft-extend, a.shareadraft-extend-cancel { display: none; }
            form.shareadraft-extend { white-space: nowrap; }
            form.shareadraft-extend, form.shareadraft-extend input, form.shareadraft-extend select {
                font-size: 11px;
            }
            th.actions, td.actions { text-align: center; }
        ';
    }

    // New method to allow draft preview
    public function allow_draft_preview($query) {
        if (!is_admin() && $query->is_main_query() 
            && isset($_GET['w321_sharedraft']) 
            && !empty($_GET['w321_sharedraft'])) {
            
            $query->set('post_status', ['publish', 'pending', 'draft', 'future', 'private']);
        }
        return $query;
    }

    // Modified to more robustly handle shared posts
    public function filter_shared_posts($posts, $query) {
        // If no posts and we have a shared draft key
        if (empty($posts) && isset($_GET['w321_sharedraft'])) {
            $key = sanitize_text_field($_GET['w321_sharedraft']);
            
            // Search through all shared drafts
            foreach ($this->admin_options as $user_shared_options) {
                if (!isset($user_shared_options['shared'])) continue;
                
                foreach ($user_shared_options['shared'] as $share) {
                    if ($share['key'] === $key) {
                        $shared_post = get_post($share['id']);
                        
                        // Verify the post exists and hasn't been published
                        if ($shared_post && $shared_post->post_status !== 'publish') {
                            return [$shared_post];
                        }
                    }
                }
            }
        }
        
        return $posts;
    }


    public function admin_page_init() {
        // jQuery is typically already included in the admin area, but we can re-ensure.
        // If not needed, remove the following line:
        // wp_enqueue_script('jquery');

        add_action('admin_head', [$this, 'print_admin_css']);
        add_action('admin_head', [$this, 'print_admin_js']);
    }

    private function get_admin_options() {
        $saved_options = get_option($this->admin_options_name);
        return is_array($saved_options) ? $saved_options : [];
    }

    private function save_admin_options() {
        $user_id = get_current_user_id();
        if ($user_id > 0) {
            $this->admin_options[$user_id] = $this->user_options;
        }
        update_option($this->admin_options_name, $this->admin_options);
    }

    private function clear_expired($all_options) {
        $all = [];
        foreach ($all_options as $user_id => $options) {
            $shared = [];
            if (!isset($options['shared']) || !is_array($options['shared'])) {
                continue;
            }
            foreach ($options['shared'] as $share) {
                if ($share['expires'] < time()) {
                    continue;
                }
                $shared[] = $share;
            }
            $options['shared'] = $shared;
            $all[$user_id] = $options;
        }
        return $all;
    }

    public function add_admin_pages() {
        add_submenu_page(
            'edit.php',
            __('Share a Draft', 'w321_sharedraft'),
            __('Share a Draft', 'w321_sharedraft'),
            'edit_posts',
            plugin_basename(__FILE__),
            [$this, 'output_existing_menu_sub_admin_page']
        );
    }

    private function calculate_seconds($params) {
        $exp = 60;
        if (isset($params['expires'])) {
            $exp = intval($params['expires']);
        }

        $mults = [
            'm' => MINUTE_IN_SECONDS,
            'h' => HOUR_IN_SECONDS,
            'd' => DAY_IN_SECONDS,
            'w' => WEEK_IN_SECONDS,
        ];

        $measure = 'w';
        if (isset($params['measure']) && isset($mults[$params['measure']])) {
            $measure = $params['measure'];
        }

        return $exp * $mults[$measure];
    }

    private function process_new_share($params) {
        $user_id = get_current_user_id();

        if (!isset($params['post_id'])) {
            return;
        }

        $post_id = absint($params['post_id']);
        $p = get_post($post_id);
        if (!$p) {
            return __('There is no such post!', 'w321_sharedraft');
        }
        if (get_post_status($p) === 'publish') {
            return __('The post is published!', 'w321_sharedraft');
        }
        if (!current_user_can('edit_post', $p->ID)) {
            return __('Sorry, you are not allowed to share posts you can’t edit.', 'w321_sharedraft');
        }

        // Use a more robust unique key generator:
        $unique_key = wp_generate_uuid4();

        $this->user_options['shared'][] = [
            'id' => $p->ID,
            'expires' => time() + $this->calculate_seconds($params),
            'key' => $unique_key,
        ];
        $this->save_admin_options();
    }

    private function process_delete($params) {
        if (!isset($params['key']) ||
            !isset($this->user_options['shared']) ||
            !is_array($this->user_options['shared'])) {
            return;
        }

        $key = sanitize_text_field($params['key']);
        $shared = [];
        foreach ($this->user_options['shared'] as $share) {
            if ($share['key'] === $key) {
                if (!current_user_can('edit_post', $share['id'])) {
                    return __('Sorry, you are not allowed to share posts you can’t edit.', 'w321_sharedraft');
                }
                // Skip adding this one (delete)
                continue;
            }
            $shared[] = $share;
        }
        $this->user_options['shared'] = $shared;
        $this->save_admin_options();
    }

    private function process_extend($params) {
        if (!isset($params['key']) ||
            !isset($this->user_options['shared']) ||
            !is_array($this->user_options['shared'])) {
            return;
        }

        $key = sanitize_text_field($params['key']);
        $shared = [];
        foreach ($this->user_options['shared'] as $share) {
            if ($share['key'] === $key) {
                if (!current_user_can('edit_post', $share['id'])) {
                    return __('Sorry, you are not allowed to share posts you can’t edit.', 'w321_sharedraft');
                }
                $share['expires'] += $this->calculate_seconds($params);
            }
            $shared[] = $share;
        }
        $this->user_options['shared'] = $shared;
        $this->save_admin_options();
    }

    private function get_drafts() {
        $user_id = get_current_user_id();
        $unpublished_statuses = ['pending', 'draft', 'future', 'private'];

        $my_unpublished = get_posts([
            'post_status'      => $unpublished_statuses,
            'author'           => $user_id,
            'suppress_filters' => false,
        ]);

        $others_unpublished = get_posts([
            'post_status'      => $unpublished_statuses,
            'author'           => -$user_id, // negative author means "any author except..."
            'suppress_filters' => false,
            'perm'             => 'editable',
        ]);

        return [
            [
                'label' => __('My unpublished posts:', 'w321_sharedraft'),
                'posts' => $my_unpublished,
            ],
            [
                'label' => __('Others’ unpublished posts:', 'w321_sharedraft'),
                'posts' => $others_unpublished,
            ],
        ];
    }

    private function get_shared() {
        if (!isset($this->user_options['shared']) || !is_array($this->user_options['shared'])) {
            return [];
        }
        return $this->user_options['shared'];
    }

    private function friendly_delta($s) {
        $m = (int)($s / MINUTE_IN_SECONDS);
        $h = (int)($s / HOUR_IN_SECONDS);
        $free_m = (int)(($s - $h * HOUR_IN_SECONDS) / MINUTE_IN_SECONDS);
        $d = (int)($s / DAY_IN_SECONDS);
        $free_h = (int)(($s - $d * DAY_IN_SECONDS) / HOUR_IN_SECONDS);

        if ($m < 1) {
            $res = [];
        } elseif ($h < 1) {
            $res = [$m];
        } elseif ($d < 1) {
            $res = [$free_m, $h];
        } else {
            $res = [$free_m, $free_h, $d];
        }

        $names = [];
        if (isset($res[0])) {
            $names[] = sprintf(_n('%d minute', '%d minutes', $res[0], 'w321_sharedraft'), $res[0]);
        }
        if (isset($res[1])) {
            $names[] = sprintf(_n('%d hour', '%d hours', $res[1], 'w321_sharedraft'), $res[1]);
        }
        if (isset($res[2])) {
            $names[] = sprintf(_n('%d day', '%d days', $res[2], 'w321_sharedraft'), $res[2]);
        }

        return implode(', ', array_reverse($names));
    }

    public function output_existing_menu_sub_admin_page() {
        $msg = '';
        if (isset($_POST['shareadraft_submit'])) {
            check_admin_referer('shareadraft-new-share');
            $msg = $this->process_new_share($_POST);
        } elseif (isset($_POST['action']) && $_POST['action'] === 'extend') {
            check_admin_referer('shareadraft-extend');
            $msg = $this->process_extend($_POST);
        } elseif (isset($_GET['action']) && $_GET['action'] === 'delete') {
            check_admin_referer('shareadraft-delete');
            $msg = $this->process_delete($_GET);
        }

        $draft_groups = $this->get_drafts();
        ?>
        <div class="wrap">
            <h2><?php _e('Share a Draft', 'w321_sharedraft'); ?></h2>
            <?php if ($msg) : ?>
                <div id="message" class="updated fade"><?php echo esc_html($msg); ?></div>
            <?php endif; ?>
            <h3><?php _e('Currently shared drafts', 'w321_sharedraft'); ?></h3>
            <table class="widefat">
                <thead>
                <tr>
                    <th><?php _e('ID', 'w321_sharedraft'); ?></th>
                    <th><?php _e('Title', 'w321_sharedraft'); ?></th>
                    <th><?php _e('Link', 'w321_sharedraft'); ?></th>
                    <th><?php _e('Expires after', 'w321_sharedraft'); ?></th>
                    <th colspan="2" class="actions"><?php _e('Actions', 'w321_sharedraft'); ?></th>
                </tr>
                </thead>
                <tbody>
                <?php
                $s = $this->get_shared();
                if (!empty($s)) {
                    foreach ($s as $share) :
                        $p = get_post($share['id']);
                        if (!$p) {
                            continue;
                        }

                        $url = add_query_arg([
                            'p' => $p->ID,
                            'w321_sharedraft' => $share['key']
                        ], home_url());

                        $friendly_delta = $this->friendly_delta($share['expires'] - time());
                        $iso_expires = date_i18n('c', $share['expires']);
                        ?>
                        <tr>
                            <td><?php echo (int)$p->ID; ?></td>
                            <td><?php echo esc_html($p->post_title); ?></td>
                            <td><a href="<?php echo esc_url($url); ?>"><?php echo esc_url($url); ?></a></td>
                            <td>
                                <time title="<?php echo esc_attr($iso_expires); ?>"
                                      datetime="<?php echo esc_attr($iso_expires); ?>">
                                    <?php echo esc_html($friendly_delta); ?>
                                </time>
                            </td>
                            <td class="actions">
                                <a class="shareadraft-extend edit"
                                   id="shareadraft-extend-link-<?php echo esc_attr($share['key']); ?>"
                                   href="javascript:shareadraft.toggle_extend('<?php echo esc_js($share['key']); ?>');">
                                    <?php _e('Extend', 'w321_sharedraft'); ?>
                                </a>
                                <form class="shareadraft-extend"
                                      id="shareadraft-extend-form-<?php echo esc_attr($share['key']); ?>"
                                      action="" method="post">
                                    <input type="hidden" name="action" value="extend"/>
                                    <input type="hidden" name="key"
                                           value="<?php echo esc_attr($share['key']); ?>"/>
                                    <input type="submit" class="button" name="shareadraft_extend_submit"
                                           value="<?php echo esc_attr__('Extend', 'w321_sharedraft'); ?>"/>
                                    <?php _e('by', 'w321_sharedraft'); ?>
                                    <?php echo $this->tmpl_measure_select(); ?>
                                    <a class="shareadraft-extend-cancel"
                                       href="javascript:shareadraft.cancel_extend('<?php echo esc_js($share['key']); ?>');">
                                        <?php _e('Cancel', 'w321_sharedraft'); ?>
                                    </a>
                                    <?php wp_nonce_field('shareadraft-extend'); ?>
                                </form>
                            </td>
                            <td class="actions">
                                <?php
                                $delete_url = add_query_arg([
                                    'page' => plugin_basename(__FILE__),
                                    'action' => 'delete',
                                    'key' => $share['key']
                                ], 'edit.php');
                                $nonced_delete_url = wp_nonce_url($delete_url, 'shareadraft-delete');
                                ?>
                                <a class="delete"
                                   href="<?php echo esc_url($nonced_delete_url); ?>"><?php _e('Delete', 'w321_sharedraft'); ?></a>
                            </td>
                        </tr>
                    <?php endforeach;
                } else {
                    ?>
                    <tr>
                        <td colspan="5"><?php _e('No shared drafts!', 'w321_sharedraft'); ?></td>
                    </tr>
                    <?php
                }
                ?>
                </tbody>
            </table>
            <h3><?php _e('Share a Draft', 'w321_sharedraft'); ?></h3>
            <form id="shareadraft-share" action="" method="post">
                <p>
                    <select id="shareadraft-postid" name="post_id">
                        <option value=""><?php _e('Choose a draft', 'w321_sharedraft'); ?></option>
                        <?php
                        foreach ($draft_groups as $draft_group) :
                            if ($draft_group['posts']) :
                                ?>
                                <option value="" disabled="disabled"></option>
                                <option value="" disabled="disabled"><?php echo esc_html($draft_group['label']); ?></option>
                                <?php
                                foreach ($draft_group['posts'] as $draft) :
                                    if (empty($draft->post_title)) {
                                        continue;
                                    }
                                    ?>
                                    <option
                                            value="<?php echo (int)$draft->ID; ?>"><?php echo esc_html($draft->post_title); ?></option>
                                <?php
                                endforeach;
                            endif;
                        endforeach;
                        ?>
                    </select>
                </p>
                <p>
                    <input type="submit" class="button" name="shareadraft_submit"
                           value="<?php echo esc_attr__('Share it', 'w321_sharedraft'); ?>"/>
                    <?php _e('for', 'w321_sharedraft'); ?>
                    <?php echo $this->tmpl_measure_select(); ?>
                </p>
                <?php wp_nonce_field('shareadraft-new-share'); ?>
            </form>
        </div>
        <?php
    }

    private function can_view($post_id) {
        if (!isset($_GET['w321_sharedraft']) || !is_array($this->admin_options)) {
            return false;
        }
        $key = sanitize_text_field($_GET['w321_sharedraft']);
        foreach ($this->admin_options as $option) {
            if (!is_array($option) || !isset($option['shared'])) {
                continue;
            }
            $shares = $option['shared'];
            foreach ($shares as $share) {
                if ((int)$share['id'] === (int)$post_id && $share['key'] === $key) {
                    return true;
                }
            }
        }
        return false;
    }

    public function posts_results_intercept($posts) {
        if (count($posts) !== 1) {
            return $posts;
        }
        $post = $posts[0];
        $status = get_post_status($post);
        if ('publish' !== $status && $this->can_view($post->ID)) {
            $this->shared_post = $post;
        }
        return $posts;
    }

    public function the_posts_intercept($posts) {
        if (empty($posts) && $this->shared_post !== null) {
            return [$this->shared_post];
        } else {
            $this->shared_post = null;
            return $posts;
        }
    }

    private function tmpl_measure_select() {
        $mins = __('minutes', 'w321_sharedraft');
        $hours = __('hours', 'w321_sharedraft');
        $days = __('days', 'w321_sharedraft');
        $weeks = __('weeks', 'w321_sharedraft');
        // default selected is weeks
        return <<<SELECT
            <input name="expires" type="text" value="2" size="4"/>
            <select name="measure">
                <option value="m">$mins</option>
                <option value="h">$hours</option>
                <option value="d">$days</option>
                <option value="w" selected>$weeks</option>
            </select>
SELECT;
    }

    public function print_admin_css() {
        ?>
        <style type="text/css">
            a.shareadraft-extend, a.shareadraft-extend-cancel { display: none; }
            form.shareadraft-extend { white-space: nowrap; }
            form.shareadraft-extend, form.shareadraft-extend input, form.shareadraft-extend select {
                font-size: 11px;
            }
            th.actions, td.actions { text-align: center; }
        </style>
        <?php
    }

    public function print_admin_js() {
        ?>
        <script type="text/javascript">
            //<![CDATA[
            (function ($) {
                $(function () {
                    $('form.shareadraft-extend').hide();
                    $('a.shareadraft-extend').show();
                    $('a.shareadraft-extend-cancel').show().css('display', 'inline');
                });
                window.shareadraft = {
                    toggle_extend: function (key) {
                        $('#shareadraft-extend-form-' + key).show();
                        $('#shareadraft-extend-link-' + key).hide();
                        $('#shareadraft-extend-form-' + key + ' input[name="expires"]').focus();
                    },
                    cancel_extend: function (key) {
                        $('#shareadraft-extend-form-' + key).hide();
                        $('#shareadraft-extend-link-' + key).show();
                    }
                };
            })(jQuery);
            //]]>
        </script>
        <?php
    }
}

endif;

if (class_exists('Share_a_Draft')) {
    new Share_a_Draft();
}