<?php
/*
Plugin Name: Bulk Term Importer from Web321
Description: Allows admins to bulk add terms to taxonomies via a textarea in the admin interface.
plugin URI: https://web321.co/plugins/bulk_taxonomy
Donate: https://www.paypal.com/paypalme/web321co/10/
Version: 1.0
Author: dewolfe001
Author URI: https://web321.co/
License: GPLv2 or later
*/

defined('ABSPATH') || exit;

define('WBT_BLOCK_NAME', plugin_basename(__FILE__));
define('WBT_BLOCK_DIR', plugin_dir_path(__FILE__));
define('WBT_BLOCK_URL', plugin_dir_url(__FILE__));
define('WBT_BLOCK_VERSION', '1.0');

add_filter('plugin_row_meta', 'wbt_block_row_meta', 10, 2);

// Donation link
function wbt_block_row_meta($links, $file) {
    if (WBT_BLOCK_NAME == $file) {
        $row_meta = [
          'donate' => '<a href="' . esc_url('https://www.paypal.com/paypalme/web321co/10/') . '" target="_blank" aria-label="' . esc_attr__('Donate', 'wbt-block') . '" >' . esc_html__('Donate', 'wbt-block') . '</a>'
        ];
        return array_merge($links, $row_meta);
    }
    return (array) $links;
}

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

// Add the action for all taxonomies
add_action('admin_init', 'wbt_add_taxonomy_actions');

function wbt_add_taxonomy_actions() {
    $taxonomies = get_taxonomies(array(), 'objects');

    foreach ($taxonomies as $taxonomy) {
        add_action("after-{$taxonomy->name}-table", 'wbt_add_textarea_to_taxonomy_pages', 10, 1);
    }
}

// Remove the meta box function as it's unnecessary
// add_action('admin_menu', 'wbt_add_meta_box');

// function wbt_add_meta_box() {
//     // This function is not needed and can be removed
// }

function wbt_add_textarea_to_taxonomy_pages($taxonomy) {
    global $pagenow;
    if ( $pagenow != 'edit-tags.php' ) {
        return;
    }
    ?>
    <h2><?php _e('Bulk Add Terms', 'bulk-term-adder'); ?></h2>
    <form method="post" action="">
        <?php wp_nonce_field('wbt_add_terms_action', 'wbt_add_terms_nonce'); ?>
        <input type="hidden" name="taxonomy" value="<?php echo esc_attr($taxonomy); ?>">
        <textarea name="wbt_terms_list" rows="10" style="width:100%;"></textarea>
        <p>
            <input type="submit" name="wbt_add_terms_submit" class="button button-primary" value="<?php _e('Add Terms', 'bulk-term-adder'); ?>">
        </p>
    </form>
    <?php
}

// Process form submission
add_action('admin_init', 'wbt_process_form_submission');

function wbt_process_form_submission() {
    if (isset($_POST['wbt_terms_list'])) {
        // Verify nonce
        check_admin_referer('wbt_add_terms_action', 'wbt_add_terms_nonce');

        // Get the current taxonomy from the posted data
        $taxonomy = sanitize_text_field($_POST['taxonomy']);

        // Ensure the user has permission to edit terms in this taxonomy
        if (!current_user_can('manage_categories')) {
            wp_die(__('You do not have sufficient permissions to access this page.', 'bulk-term-adder'));
        }

        // Get the list of terms
        $terms_list = sanitize_textarea_field($_POST['wbt_terms_list']);
        $terms = array_filter(array_map('trim', explode("\n", $terms_list)));

        $added = 0;
        $skipped = 0;

        foreach ($terms as $term_name) {
            if (!term_exists($term_name, $taxonomy)) {
                wp_insert_term($term_name, $taxonomy);
                $added++;
            } else {
                $skipped++;
            }
        }

        // Store the counts in a transient to display after redirect
        set_transient('wbt_term_counts', array('added' => $added, 'skipped' => $skipped), 30);

        // Redirect back to the taxonomy page
        wp_safe_redirect(admin_url("edit-tags.php?taxonomy={$taxonomy}"));
        exit;
    }
}

// Display an admin notice with counts
add_action('admin_notices', 'wbt_display_admin_notice');

function wbt_display_admin_notice() {
    if ($counts = get_transient('wbt_term_counts')) {
        delete_transient('wbt_term_counts');
        ?>
        <div class="notice notice-success is-dismissible">
            <p><?php printf(
                __('Terms added: %d. Terms skipped (already existed): %d.', 'bulk-term-adder'),
                $counts['added'],
                $counts['skipped']
            ); ?></p>
        </div>
        <?php
    }
}
