<?php
/**
 * Plugin Name: Quick Contact Buttons
 * Plugin URI: https://web321.ca
 * Description: Adds quick access contact buttons (phone, SMS, email) on the right side of the screen
 * Version: 1.0.0
 * Author: Web321 Marketing Ltd. (dewolfe001)
 * Author URI: https://web321.co
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: quick-contact-buttons
 */


// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('QCB_VERSION', '1.0.0');
define('QCB_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('QCB_PLUGIN_URL', plugin_dir_url(__FILE__));
define('QCB_BLOCK_NAME', plugin_basename(__FILE__));

add_filter('plugin_row_meta', 'qcb_block_row_meta', 10, 2);

// Donation link
function qcb_block_row_meta($links, $file) {
    if (QCB_BLOCK_NAME == $file) {
        $row_meta = [
          'settings' => '<a href="' . esc_url('/wp-admin/options-general.php?page=quick-contact-buttons') . '" target="_blank" aria-label="' . esc_attr__('Settings', 'wvr-block') . '" >' . esc_html__('Settings', 'pwc-block') . '</a>',
          'donate' => '<a href="' . esc_url('https://www.paypal.com/paypalme/web321co/10/') . '" target="_blank" aria-label="' . esc_attr__('Donate', 'wvr-block') . '" >' . esc_html__('Donate', 'pwc-block') . '</a>',
        ];
        return array_merge($links, $row_meta);
    }
    return (array) $links;
}

class Quick_Contact_Buttons {
    
    private static $instance = null;
    
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        // Admin hooks
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'register_settings'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_assets'));
        
        // Frontend hooks
        add_action('wp_enqueue_scripts', array($this, 'enqueue_frontend_assets'));
        add_action('wp_footer', array($this, 'render_contact_buttons'));
    }
    
    /**
     * Add admin menu page
     */
    public function add_admin_menu() {
        add_options_page(
            'Quick Contact Buttons Settings',
            'Contact Buttons',
            'manage_options',
            'quick-contact-buttons',
            array($this, 'render_admin_page')
        );
    }
    
    /**
     * Register plugin settings
     */
    public function register_settings() {
        register_setting('qcb_settings_group', 'qcb_phone_number');
        register_setting('qcb_settings_group', 'qcb_sms_number');
        register_setting('qcb_settings_group', 'qcb_email');
        register_setting('qcb_settings_group', 'qcb_bg_color', array(
            'default' => '#0073aa'
        ));
        register_setting('qcb_settings_group', 'qcb_fg_color', array(
            'default' => '#ffffff'
        ));
    }
    
    /**
     * Enqueue admin assets
     */
    public function enqueue_admin_assets($hook) {
        if ('settings_page_quick-contact-buttons' !== $hook) {
            return;
        }
        
        // Add WordPress color picker
        wp_enqueue_style('wp-color-picker');
        wp_enqueue_script('wp-color-picker');
        
        wp_enqueue_style(
            'qcb-admin-css',
            QCB_PLUGIN_URL . 'assets/css/admin.css',
            array(),
            QCB_VERSION
        );
        
        // Inline script to initialize color pickers
        wp_add_inline_script('wp-color-picker', '
            jQuery(document).ready(function($) {
                $(".fcb-color-picker").wpColorPicker();
            });
        ');
    }
    
    /**
     * Enqueue frontend assets
     */
    public function enqueue_frontend_assets() {
        wp_enqueue_style(
            'qcb-frontend-css',
            QCB_PLUGIN_URL . 'assets/css/frontend.css',
            array(),
            QCB_VERSION
        );
        
        wp_enqueue_script(
            'qcb-frontend-js',
            QCB_PLUGIN_URL . 'assets/js/frontend.js',
            array('jquery'),
            QCB_VERSION,
            true
        );
    }
    
    /**
     * Render admin settings page
     */
    public function render_admin_page() {
        if (!current_user_can('manage_options')) {
            return;
        }
        
        include QCB_PLUGIN_DIR . 'admin/settings-page.php';
    }
    
    /**
     * Render contact buttons in footer
     */
    public function render_contact_buttons() {
        $phone = get_option('qcb_phone_number', '');
        $sms = get_option('qcb_sms_number', '');
        $email = get_option('qcb_email', '');
        $bg_color = get_option('qcb_bg_color', '#0073aa');
        $fg_color = get_option('qcb_fg_color', '#ffffff');
        
        // Don't render if all fields are empty
        if (empty($phone) && empty($sms) && empty($email)) {
            return;
        }
        ?>
        <div id="qcb-container" class="qcb-hidden" style="--qcb-bg-color: <?php echo esc_attr($bg_color); ?>; --qcb-fg-color: <?php echo esc_attr($fg_color); ?>;">
            <div id="qcb-buttons">
                <?php if (!empty($phone)) : ?>
                <a href="tel:<?php echo esc_attr($phone); ?>" class="qcb-button qcb-phone" aria-label="Call us">
                    <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="currentColor" width="24" height="24">
                        <path d="M20.01 15.38c-1.23 0-2.42-.2-3.53-.56-.35-.12-.74-.03-1.01.24l-1.57 1.97c-2.83-1.35-5.48-3.9-6.89-6.83l1.95-1.66c.27-.28.35-.67.24-1.02-.37-1.11-.56-2.3-.56-3.53 0-.54-.45-.99-.99-.99H4.19C3.65 3 3 3.24 3 3.99 3 13.28 10.73 21 20.01 21c.71 0 .99-.63.99-1.18v-3.45c0-.54-.45-.99-.99-.99z"/>
                    </svg>
                </a>
                <?php endif; ?>
                
                <?php if (!empty($sms)) : ?>
                <a href="sms:<?php echo esc_attr($sms); ?>" class="qcb-button qcb-sms" aria-label="Send SMS">
                    <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="currentColor" width="24" height="24">
                        <path d="M20 2H4c-1.1 0-1.99.9-1.99 2L2 22l4-4h14c1.1 0 2-.9 2-2V4c0-1.1-.9-2-2-2zM9 11H7V9h2v2zm4 0h-2V9h2v2zm4 0h-2V9h2v2z"/>
                    </svg>
                </a>
                <?php endif; ?>
                
                <?php if (!empty($email)) : ?>
                <a href="mailto:<?php echo esc_attr($email); ?>" class="qcb-button qcb-email" aria-label="Email us">
                    <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="currentColor" width="24" height="24">
                        <path d="M20 4H4c-1.1 0-1.99.9-1.99 2L2 18c0 1.1.9 2 2 2h16c1.1 0 2-.9 2-2V6c0-1.1-.9-2-2-2zm0 4l-8 5-8-5V6l8 5 8-5v2z"/>
                    </svg>
                </a>
                <?php endif; ?>
            </div>
            
            <button id="qcb-toggle" class="qcb-toggle" aria-label="Toggle contact buttons">
                <svg class="qcb-close-icon" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="currentColor" width="24" height="24">
                    <path d="M19 6.41L17.59 5 12 10.59 6.41 5 5 6.41 10.59 12 5 17.59 6.41 19 12 13.41 17.59 19 19 17.59 13.41 12z"/>
                </svg>
                <svg class="qcb-open-icon" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="currentColor" width="24" height="24">
                    <path d="M19 13h-6v6h-2v-6H5v-2h6V5h2v6h6v2z"/>
                </svg>
            </button>
        </div>
        <?php
    }
}

// Initialize the plugin
Quick_Contact_Buttons::get_instance();
