<?php
/**
 * Plugin Name:       NextGen Vertical Filmstrip (Standalone)
 * Plugin URI:        https://shawndewolfe.com/ngvf-standalone
 * Description:       A standalone vertical filmstrip gallery that uses NextGen Gallery images. Bypasses NextGen's display type system for better reliability.
 * Version:           2.3.0
 * Author:            dewolfe001
 * Author URI:        https://shawndewolfe.com
 * License:           GPL-2.0+
 * License URI:       http://www.gnu.org/licenses/gpl-2.0.txt
 * Text Domain:       ngvf-standalone
 * Domain Path:       /languages
 */

// If this file is called directly, abort.
if (!defined('WPINC')) {
    die;
}

define('NGVF_SA_VERSION', '2.3.0');
define('NGVF_SA_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('NGVF_SA_PLUGIN_URL', plugin_dir_url(__FILE__));

class NGVF_Standalone {
    
    protected static $instance = null;
    
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        add_action('init', array($this, 'init'));
        
        // Add plugin page links
        add_filter('plugin_row_meta', array($this, 'plugin_row_meta'), 10, 2);
        add_filter('plugin_action_links_' . plugin_basename(__FILE__), array($this, 'plugin_action_links'));
    }
    
    public function init() {
        // Register shortcode
        add_shortcode('ngvf_gallery', array($this, 'render_gallery'));
        
        // Enqueue scripts and styles
        add_action('wp_enqueue_scripts', array($this, 'enqueue_assets'));
        
        // Add admin menu and plugin details
        if (is_admin()) {
            add_action('admin_menu', array($this, 'add_settings_page'));
            add_action('admin_init', array($this, 'register_settings'));
            add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_assets'));
            add_action('admin_footer', array($this, 'plugin_details_modal'));
        }
    }
    
    /**
     * Add donate and documentation links to plugin row meta
     */
    public function plugin_row_meta($links, $file) {
        if (plugin_basename(__FILE__) === $file) {
            $row_meta = array(
                'donate' => '<a href="' . esc_url('https://www.paypal.com/paypalme/web321co/20/') . '" target="_blank" aria-label="' . esc_attr__('Donate', 'ngvf-standalone') . '" style="color: #d54e21; font-weight: 600;">' . esc_html__('♥ Donate', 'ngvf-standalone') . '</a>',
                'docs' => '<a href="' . esc_url('https://shawndewolfe.com/ngvf-standalone-docs/') . '" target="_blank" aria-label="' . esc_attr__('Documentation', 'ngvf-standalone') . '">' . esc_html__('Documentation', 'ngvf-standalone') . '</a>',
                'support' => '<a href="' . esc_url('https://shawndewolfe.com/support/') . '" target="_blank" aria-label="' . esc_attr__('Support', 'ngvf-standalone') . '">' . esc_html__('Support', 'ngvf-standalone') . '</a>'
            );
            return array_merge($links, $row_meta);
        }
        return (array) $links;
    }
    
    /**
     * Add settings link to plugin action links
     */
    public function plugin_action_links($links) {
        $settings_link = '<a href="' . esc_url(admin_url('options-general.php?page=ngvf-settings')) . '">' . esc_html__('Settings', 'ngvf-standalone') . '</a>';
        array_unshift($links, $settings_link);
        return $links;
    }
    
    /**
     * Add settings page to admin menu
     */
    public function add_settings_page() {
        add_options_page(
            __('NGVF Gallery Settings', 'ngvf-standalone'),
            __('NGVF Gallery', 'ngvf-standalone'),
            'manage_options',
            'ngvf-settings',
            array($this, 'render_settings_page')
        );
    }
    
    /**
     * Register plugin settings
     */
    public function register_settings() {
        register_setting('ngvf_settings', 'ngvf_default_columns');
        register_setting('ngvf_settings', 'ngvf_default_per_page');
        register_setting('ngvf_settings', 'ngvf_default_thumb_width');
        register_setting('ngvf_settings', 'ngvf_default_thumb_height');
        register_setting('ngvf_settings', 'ngvf_default_lightbox');
        register_setting('ngvf_settings', 'ngvf_default_pagination');
        
        add_settings_section(
            'ngvf_defaults_section',
            __('Default Gallery Settings', 'ngvf-standalone'),
            array($this, 'settings_section_callback'),
            'ngvf-settings'
        );
        
        add_settings_field(
            'ngvf_default_columns',
            __('Default Columns', 'ngvf-standalone'),
            array($this, 'columns_field_callback'),
            'ngvf-settings',
            'ngvf_defaults_section'
        );
        
        add_settings_field(
            'ngvf_default_per_page',
            __('Default Images Per Page', 'ngvf-standalone'),
            array($this, 'per_page_field_callback'),
            'ngvf-settings',
            'ngvf_defaults_section'
        );
        
        add_settings_field(
            'ngvf_default_thumb_width',
            __('Default Thumbnail Width', 'ngvf-standalone'),
            array($this, 'thumb_width_field_callback'),
            'ngvf-settings',
            'ngvf_defaults_section'
        );
        
        add_settings_field(
            'ngvf_default_thumb_height',
            __('Default Thumbnail Height', 'ngvf-standalone'),
            array($this, 'thumb_height_field_callback'),
            'ngvf-settings',
            'ngvf_defaults_section'
        );
        
        add_settings_field(
            'ngvf_default_lightbox',
            __('Enable Lightbox by Default', 'ngvf-standalone'),
            array($this, 'lightbox_field_callback'),
            'ngvf-settings',
            'ngvf_defaults_section'
        );
        
        add_settings_field(
            'ngvf_default_pagination',
            __('Enable Pagination by Default', 'ngvf-standalone'),
            array($this, 'pagination_field_callback'),
            'ngvf-settings',
            'ngvf_defaults_section'
        );
    }
    
    /**
     * Settings section callback
     */
    public function settings_section_callback() {
        echo '<p>' . esc_html__('Set default values for your galleries. These can be overridden in individual shortcodes.', 'ngvf-standalone') . '</p>';
    }
    
    /**
     * Columns field callback
     */
    public function columns_field_callback() {
        $value = get_option('ngvf_default_columns', 3);
        echo '<input type="number" name="ngvf_default_columns" value="' . esc_attr($value) . '" min="1" max="4" class="small-text" />';
        echo '<p class="description">' . esc_html__('Number of thumbnail columns (1-4)', 'ngvf-standalone') . '</p>';
    }
    
    /**
     * Per page field callback
     */
    public function per_page_field_callback() {
        $value = get_option('ngvf_default_per_page', 20);
        echo '<input type="number" name="ngvf_default_per_page" value="' . esc_attr($value) . '" min="1" class="small-text" />';
        echo '<p class="description">' . esc_html__('Number of images to display per page', 'ngvf-standalone') . '</p>';
    }
    
    /**
     * Thumb width field callback
     */
    public function thumb_width_field_callback() {
        $value = get_option('ngvf_default_thumb_width', 150);
        echo '<input type="number" name="ngvf_default_thumb_width" value="' . esc_attr($value) . '" min="50" class="small-text" />';
        echo '<p class="description">' . esc_html__('Thumbnail width in pixels', 'ngvf-standalone') . '</p>';
    }
    
    /**
     * Thumb height field callback
     */
    public function thumb_height_field_callback() {
        $value = get_option('ngvf_default_thumb_height', 150);
        echo '<input type="number" name="ngvf_default_thumb_height" value="' . esc_attr($value) . '" min="50" class="small-text" />';
        echo '<p class="description">' . esc_html__('Thumbnail height in pixels', 'ngvf-standalone') . '</p>';
    }
    
    /**
     * Lightbox field callback
     */
    public function lightbox_field_callback() {
        $value = get_option('ngvf_default_lightbox', 'true');
        echo '<input type="checkbox" name="ngvf_default_lightbox" value="true" ' . checked($value, 'true', false) . ' />';
        echo '<p class="description">' . esc_html__('Enable lightbox popup for images', 'ngvf-standalone') . '</p>';
    }
    
    /**
     * Pagination field callback
     */
    public function pagination_field_callback() {
        $value = get_option('ngvf_default_pagination', 'true');
        echo '<input type="checkbox" name="ngvf_default_pagination" value="true" ' . checked($value, 'true', false) . ' />';
        echo '<p class="description">' . esc_html__('Show pagination controls', 'ngvf-standalone') . '</p>';
    }
    
    /**
     * Render settings page
     */
    public function render_settings_page() {
        if (!current_user_can('manage_options')) {
            return;
        }
        
        // Save success message
        if (isset($_GET['settings-updated'])) {
            add_settings_error('ngvf_messages', 'ngvf_message', __('Settings Saved', 'ngvf-standalone'), 'updated');
        }
        
        settings_errors('ngvf_messages');
        ?>
        <div class="wrap">
            <h1><?php echo esc_html(get_admin_page_title()); ?></h1>
            
            <div style="background: #fff; border: 1px solid #ccc; padding: 20px; margin: 20px 0; border-radius: 4px;">
                <h2><?php esc_html_e('How to Use', 'ngvf-standalone'); ?></h2>
                <p><?php esc_html_e('Add a gallery to any page or post using the shortcode:', 'ngvf-standalone'); ?></p>
                <code style="background: #f5f5f5; padding: 10px; display: block; margin: 10px 0;">[ngvf_gallery id="1"]</code>
                
                <h3><?php esc_html_e('With Custom Options:', 'ngvf-standalone'); ?></h3>
                <code style="background: #f5f5f5; padding: 10px; display: block; margin: 10px 0;">[ngvf_gallery id="1" columns="3" per_page="20" lightbox="true"]</code>
                
                <h3><?php esc_html_e('Available Parameters:', 'ngvf-standalone'); ?></h3>
                <ul style="list-style: disc; margin-left: 20px;">
                    <li><strong>id</strong> - Gallery ID (required)</li>
                    <li><strong>columns</strong> - Number of thumbnail columns (1-4)</li>
                    <li><strong>per_page</strong> - Images per page</li>
                    <li><strong>thumb_width</strong> - Thumbnail width in pixels</li>
                    <li><strong>thumb_height</strong> - Thumbnail height in pixels</li>
                    <li><strong>lightbox</strong> - Enable/disable lightbox (true/false)</li>
                    <li><strong>pagination</strong> - Enable/disable pagination (true/false)</li>
                </ul>
                
                <p style="margin-top: 20px;">
                    <strong><?php esc_html_e('Find Gallery ID:', 'ngvf-standalone'); ?></strong> 
                    <?php esc_html_e('Go to Gallery → Manage Galleries and note the ID number.', 'ngvf-standalone'); ?>
                </p>
            </div>
            
            <form action="options.php" method="post">
                <?php
                settings_fields('ngvf_settings');
                do_settings_sections('ngvf-settings');
                submit_button(__('Save Settings', 'ngvf-standalone'));
                ?>
            </form>
            
            <div style="background: #f0f8ff; border: 1px solid #0073aa; padding: 20px; margin: 20px 0; border-radius: 4px;">
                <h2 style="margin-top: 0;">♥ <?php esc_html_e('Support This Plugin', 'ngvf-standalone'); ?></h2>
                <p><?php esc_html_e('If you find this plugin helpful, please consider making a donation to support continued development.', 'ngvf-standalone'); ?></p>
                <p>
                    <a href="https://www.paypal.com/paypalme/web321co/20/" target="_blank" class="button button-primary" style="background: #d54e21; border-color: #d54e21; text-shadow: none;">
                        <?php esc_html_e('Donate via PayPal', 'ngvf-standalone'); ?>
                    </a>
                </p>
                <p>
                    <strong><?php esc_html_e('Need Help?', 'ngvf-standalone'); ?></strong><br>
                    <a href="https://shawndewolfe.com/support/" target="_blank"><?php esc_html_e('Visit Support Page', 'ngvf-standalone'); ?></a> | 
                    <a href="https://shawndewolfe.com/ngvf-standalone-docs/" target="_blank"><?php esc_html_e('Read Documentation', 'ngvf-standalone'); ?></a>
                </p>
            </div>
        </div>
        <?php
    }
    
    public function enqueue_assets() {
        // Enqueue lightbox CSS
        wp_enqueue_style(
            'ngvf-lightbox',
            NGVF_SA_PLUGIN_URL . 'assets/css/lightbox.css',
            array(),
            NGVF_SA_VERSION
        );
        
        wp_enqueue_style(
            'ngvf-standalone-style',
            NGVF_SA_PLUGIN_URL . 'assets/css/style.css',
            array('ngvf-lightbox'),
            NGVF_SA_VERSION
        );
        
        // Enqueue lightbox JS
        wp_enqueue_script(
            'ngvf-lightbox',
            NGVF_SA_PLUGIN_URL . 'assets/js/lightbox.js',
            array('jquery'),
            NGVF_SA_VERSION,
            true
        );
        
        wp_enqueue_script(
            'ngvf-standalone-script',
            NGVF_SA_PLUGIN_URL . 'assets/js/script.js',
            array('jquery', 'ngvf-lightbox'),
            NGVF_SA_VERSION,
            true
        );
    }
    
    /**
     * Enqueue admin assets
     */
    public function enqueue_admin_assets($hook) {
        // Only load on plugins page
        if ($hook !== 'plugins.php') {
            return;
        }
        
        wp_enqueue_style('thickbox');
        wp_enqueue_script('thickbox');
        
        // Add custom admin CSS
        wp_add_inline_style('thickbox', '
            #ngvf-plugin-details { display: none; }
            #TB_window.ngvf-details { width: 800px !important; margin-left: -400px !important; }
            #TB_window.ngvf-details #TB_ajaxContent { width: 100% !important; padding: 20px; }
            .ngvf-details-content h2 { margin-top: 0; color: #23282d; border-bottom: 2px solid #0073aa; padding-bottom: 10px; }
            .ngvf-details-content h3 { color: #0073aa; margin-top: 25px; }
            .ngvf-details-content code { background: #f5f5f5; padding: 2px 6px; border-radius: 3px; }
            .ngvf-details-content pre { background: #f5f5f5; padding: 15px; border-radius: 4px; overflow-x: auto; }
            .ngvf-details-content .feature-list { list-style: none; padding: 0; }
            .ngvf-details-content .feature-list li { padding: 8px 0; padding-left: 25px; position: relative; }
            .ngvf-details-content .feature-list li:before { content: "✓"; position: absolute; left: 0; color: #46b450; font-weight: bold; }
            .ngvf-details-content .info-box { background: #f0f8ff; border-left: 4px solid #0073aa; padding: 15px; margin: 20px 0; }
            .ngvf-details-content .warning-box { background: #fff8e5; border-left: 4px solid #f0b849; padding: 15px; margin: 20px 0; }
            .ngvf-details-content .donate-box { background: #f0fff4; border: 2px solid #46b450; padding: 20px; margin: 20px 0; text-align: center; border-radius: 8px; }
            .ngvf-details-content .donate-box .button { background: #d54e21; border-color: #d54e21; color: white; padding: 10px 30px; height: auto; font-size: 16px; }
            .ngvf-details-content table { width: 100%; border-collapse: collapse; margin: 15px 0; }
            .ngvf-details-content table th { background: #0073aa; color: white; padding: 10px; text-align: left; }
            .ngvf-details-content table td { border: 1px solid #ddd; padding: 10px; }
            .ngvf-details-content table tr:nth-child(even) { background: #f9f9f9; }
        ');
        
        // Add JavaScript to handle the plugin details link
        wp_add_inline_script('thickbox', '
            jQuery(document).ready(function($) {
                // Intercept plugin details link
                $("tr[data-plugin=\'ngvf-standalone/ngvf-standalone.php\'] .plugin-title a").on("click", function(e) {
                    var href = $(this).attr("href");
                    if (href && href.indexOf("plugin-information") > -1) {
                        e.preventDefault();
                        tb_show("Plugin Details: NextGen Vertical Filmstrip (Standalone)", "#TB_inline?width=800&height=600&inlineId=ngvf-plugin-details", "");
                        $("#TB_window").addClass("ngvf-details");
                    }
                });
            });
        ');
    }
    
    /**
     * Display plugin details modal
     */
    public function plugin_details_modal() {
        $screen = get_current_screen();
        if ($screen->id !== 'plugins') {
            return;
        }
        ?>
        <div id="ngvf-plugin-details" style="display:none;">
            <div class="ngvf-details-content">
                <h2>NextGen Vertical Filmstrip (Standalone)</h2>
                
                <p><strong>Version:</strong> <?php echo NGVF_SA_VERSION; ?> | <strong>Author:</strong> dewolfe001 | <strong>License:</strong> GPL-2.0+</p>
                
                <div class="info-box">
                    <strong>A reliable, standalone gallery plugin that uses NextGen Gallery's images without depending on NextGen's buggy display type system.</strong>
                </div>
                
                <h3>✨ Key Features</h3>
                <ul class="feature-list">
                    <li>Vertical filmstrip layout with thumbnails on left</li>
                    <li>Beautiful fullscreen lightbox with navigation</li>
                    <li>Settings page for configuring defaults</li>
                    <li>Works around NextGen Gallery SQL bugs</li>
                    <li>Simple shortcode: <code>[ngvf_gallery id="1"]</code></li>
                    <li>Responsive design for all devices</li>
                    <li>Keyboard navigation (arrow keys, ESC)</li>
                    <li>Touch-friendly mobile interface</li>
                    <li>No external dependencies</li>
                </ul>
                
                <h3>📖 Usage</h3>
                
                <h4>Basic Shortcode</h4>
                <pre><code>[ngvf_gallery id="1"]</code></pre>
                
                <h4>With Options</h4>
                <pre><code>[ngvf_gallery id="1" columns="3" per_page="20" lightbox="true"]</code></pre>
                
                <h4>Available Parameters</h4>
                <table>
                    <thead>
                        <tr>
                            <th>Parameter</th>
                            <th>Default</th>
                            <th>Description</th>
                        </tr>
                    </thead>
                    <tbody>
                        <tr>
                            <td><code>id</code></td>
                            <td>Required</td>
                            <td>NextGen Gallery ID</td>
                        </tr>
                        <tr>
                            <td><code>columns</code></td>
                            <td>3</td>
                            <td>Thumbnail columns (1-4)</td>
                        </tr>
                        <tr>
                            <td><code>per_page</code></td>
                            <td>20</td>
                            <td>Images per page</td>
                        </tr>
                        <tr>
                            <td><code>thumb_width</code></td>
                            <td>150</td>
                            <td>Thumbnail width in pixels</td>
                        </tr>
                        <tr>
                            <td><code>thumb_height</code></td>
                            <td>150</td>
                            <td>Thumbnail height in pixels</td>
                        </tr>
                        <tr>
                            <td><code>lightbox</code></td>
                            <td>true</td>
                            <td>Enable lightbox (true/false)</td>
                        </tr>
                        <tr>
                            <td><code>pagination</code></td>
                            <td>true</td>
                            <td>Show pagination (true/false)</td>
                        </tr>
                    </tbody>
                </table>
                
                <h3>⚙️ Settings Page</h3>
                <p>Configure default values at <strong>Settings → NGVF Gallery</strong></p>
                <ul>
                    <li>Set defaults for all galleries</li>
                    <li>Use simple shortcodes without repeating attributes</li>
                    <li>Override per gallery when needed</li>
                    <li>Built-in usage instructions and examples</li>
                </ul>
                
                <h3>🔍 Finding Your Gallery ID</h3>
                <ol>
                    <li>Go to <strong>Gallery → Manage Galleries</strong></li>
                    <li>Look at the first column for the gallery ID</li>
                    <li>Use that number in your shortcode: <code>[ngvf_gallery id="YOUR_ID"]</code></li>
                </ol>
                
                <h3>💡 Why This Plugin?</h3>
                <div class="warning-box">
                    <p><strong>Problem:</strong> NextGen Gallery has SQL syntax errors that prevent custom display types from working.</p>
                    <p><strong>Solution:</strong> This plugin bypasses NextGen's broken display type system and accesses gallery data directly from the database.</p>
                </div>
                
                <h3>🎯 Advantages</h3>
                <ul class="feature-list">
                    <li>Works even when NextGen has SQL errors</li>
                    <li>More reliable than NextGen display types</li>
                    <li>Direct database access for better performance</li>
                    <li>Simple, memorable shortcode syntax</li>
                    <li>Independent of NextGen's internal updates</li>
                    <li>Professional settings page</li>
                </ul>
                
                <h3>📚 Documentation</h3>
                <p>Complete documentation is included in the plugin folder:</p>
                <ul>
                    <li><strong>README.md</strong> - Complete feature documentation</li>
                    <li><strong>QUICKSTART.md</strong> - Quick setup guide</li>
                    <li><strong>CHANGELOG.md</strong> - Version history</li>
                </ul>
                
                <h3>🛠️ Support</h3>
                <p>Need help? Visit: <a href="https://shawndewolfe.com/support/" target="_blank">https://shawndewolfe.com/support/</a></p>
                <p>Full documentation: <a href="https://shawndewolfe.com/ngvf-standalone-docs/" target="_blank">https://shawndewolfe.com/ngvf-standalone-docs/</a></p>
                
                <div class="donate-box">
                    <h3 style="margin-top: 0;">♥ Support This Plugin</h3>
                    <p>If you find this plugin helpful, please consider making a donation to support continued development.</p>
                    <p>
                        <a href="https://www.paypal.com/paypalme/web321co/20/" target="_blank" class="button button-primary">
                            Donate via PayPal
                        </a>
                    </p>
                    <p style="margin-bottom: 0;"><small>Your support helps keep this plugin free and actively maintained!</small></p>
                </div>
                
                <h3>📝 Quick Examples</h3>
                
                <h4>2 Columns with 12 Images</h4>
                <pre><code>[ngvf_gallery id="1" columns="2" per_page="12"]</code></pre>
                
                <h4>4 Columns with Large Thumbnails</h4>
                <pre><code>[ngvf_gallery id="2" columns="4" thumb_width="200" thumb_height="200"]</code></pre>
                
                <h4>Disable Lightbox</h4>
                <pre><code>[ngvf_gallery id="3" lightbox="false"]</code></pre>
                
                <h4>Show All Images (No Pagination)</h4>
                <pre><code>[ngvf_gallery id="4" per_page="1000" pagination="false"]</code></pre>
                
                <h3>✅ Requirements</h3>
                <ul>
                    <li>WordPress 5.0 or higher</li>
                    <li>NextGen Gallery (any version)</li>
                    <li>PHP 7.4 or higher</li>
                </ul>
                
                <h3>🎉 What's New in Version <?php echo NGVF_SA_VERSION; ?></h3>
                <ul class="feature-list">
                    <li>Settings page for default configuration</li>
                    <li>Donate link on plugins page</li>
                    <li>Quick access links (Settings, Docs, Support)</li>
                    <li>Built-in help and documentation</li>
                    <li>Full lightbox functionality with keyboard navigation</li>
                    <li>Professional admin interface</li>
                </ul>
                
                <p style="text-align: center; margin-top: 30px;">
                    <strong>Made with ❤️ by dewolfe001</strong><br>
                    <a href="https://shawndewolfe.com" target="_blank">https://shawndewolfe.com</a>
                </p>
            </div>
        </div>
        <?php
    }
    
    /**
     * Render the gallery shortcode
     * 
     * @param array $atts Shortcode attributes
     * @return string Gallery HTML
     */
    public function render_gallery($atts) {
        // Get default values from settings
        $default_columns = get_option('ngvf_default_columns', 3);
        $default_per_page = get_option('ngvf_default_per_page', 20);
        $default_thumb_width = get_option('ngvf_default_thumb_width', 150);
        $default_thumb_height = get_option('ngvf_default_thumb_height', 150);
        $default_lightbox = get_option('ngvf_default_lightbox', 'true');
        $default_pagination = get_option('ngvf_default_pagination', 'true');
        
        // Parse attributes with defaults from settings
        $atts = shortcode_atts(array(
            'id' => 0,
            'columns' => $default_columns,
            'per_page' => $default_per_page,
            'thumb_width' => $default_thumb_width,
            'thumb_height' => $default_thumb_height,
            'lightbox' => $default_lightbox,
            'pagination' => $default_pagination
        ), $atts, 'ngvf_gallery');
        
        $gallery_id = intval($atts['id']);
        
        if ($gallery_id === 0) {
            return '<p style="color: red;">Error: Gallery ID is required. Use [ngvf_gallery id="1"]</p>';
        }
        
        // Get gallery images
        $images = $this->get_gallery_images($gallery_id);
        
        if (empty($images)) {
            return '<p>No images found in this gallery.</p>';
        }
        
        // Apply pagination
        $images_per_page = intval($atts['per_page']);
        $current_page = isset($_GET['ngvf_page']) ? intval($_GET['ngvf_page']) : 1;
        $offset = ($current_page - 1) * $images_per_page;
        
        $total_images = count($images);
        $total_pages = ceil($total_images / $images_per_page);
        $paginated_images = array_slice($images, $offset, $images_per_page);
        
        // Generate unique ID
        $unique_id = 'ngvf-gallery-' . $gallery_id . '-' . uniqid();
        
        // Start output
        ob_start();
        include NGVF_SA_PLUGIN_DIR . 'templates/gallery.php';
        return ob_get_clean();
    }
    
    /**
     * Get images from a NextGen Gallery
     * 
     * @param int $gallery_id Gallery ID
     * @return array Array of image objects
     */
    private function get_gallery_images($gallery_id) {
        global $wpdb;
        
        // Try to use NextGen Gallery's storage system if available
        if (class_exists('C_Gallery_Storage')) {
            $storage = C_Gallery_Storage::get_instance();
            
            // Query NextGen's image table
            $table_name = $wpdb->prefix . 'ngg_pictures';
            $images = $wpdb->get_results($wpdb->prepare(
                "SELECT * FROM {$table_name} WHERE galleryid = %d AND exclude = 0 ORDER BY sortorder, pid ASC",
                $gallery_id
            ));
            
            if (!empty($images)) {
                // Add full URLs to each image
                foreach ($images as &$image) {
                    $image->thumb_url = $storage->get_thumb_url($image);
                    $image->image_url = $storage->get_image_url($image);
                    $image->full_url = $storage->get_full_url($image);
                }
            }
            
            return $images;
        }
        
        // Fallback: Manual URL construction if NextGen classes aren't available
        $table_name = $wpdb->prefix . 'ngg_pictures';
        $gallery_table = $wpdb->prefix . 'ngg_gallery';
        
        $images = $wpdb->get_results($wpdb->prepare(
            "SELECT p.*, g.path 
             FROM {$table_name} p 
             LEFT JOIN {$gallery_table} g ON p.galleryid = g.gid 
             WHERE p.galleryid = %d AND p.exclude = 0 
             ORDER BY p.sortorder, p.pid ASC",
            $gallery_id
        ));
        
        if (!empty($images)) {
            $upload_dir = wp_upload_dir();
            $base_url = $upload_dir['baseurl'];
            
            foreach ($images as &$image) {
                $gallery_path = trailingslashit($image->path);
                
                // Construct URLs manually
                $image->thumb_url = $base_url . '/ngg_thumbnails/' . $image->filename;
                $image->image_url = $base_url . '/' . $gallery_path . $image->filename;
                $image->full_url = $image->image_url;
                
                // Try thumbs subfolder if main thumbnails don't exist
                if (!file_exists(str_replace($base_url, $upload_dir['basedir'], $image->thumb_url))) {
                    $image->thumb_url = $base_url . '/' . $gallery_path . 'thumbs/thumbs_' . $image->filename;
                }
            }
        }
        
        return $images ? $images : array();
    }
}

// Initialize the plugin
function ngvf_standalone_init() {
    return NGVF_Standalone::get_instance();
}
add_action('plugins_loaded', 'ngvf_standalone_init');
