/**
 * NGVF Standalone Gallery JavaScript
 * 
 * @package NGVF_Standalone
 * @version 2.0.0
 */

(function($) {
    'use strict';
    
    $(document).ready(function() {
        initGalleries();
    });
    
    function initGalleries() {
        $('.ngvf-standalone-gallery').each(function() {
            var $gallery = $(this);
            initThumbnailClicks($gallery);
            initKeyboardNav($gallery);
        });
    }
    
    /**
     * Initialize thumbnail click handlers
     */
    function initThumbnailClicks($gallery) {
        var $thumbnails = $gallery.find('.ngvf-thumb');
        var $mainImg = $gallery.find('.ngvf-main-img');
        var $mainLink = $gallery.find('.ngvf-main-link');
        var $description = $gallery.find('.ngvf-description');
        var $mainWrapper = $gallery.find('.ngvf-main-wrapper');
        
        $thumbnails.on('click keypress', function(e) {
            if (e.type === 'keypress' && e.which !== 13 && e.which !== 32) {
                return;
            }
            
            e.preventDefault();
            
            var $thumb = $(this);
            
            if ($thumb.hasClass('active')) {
                return;
            }
            
            // Get image data
            var fullUrl = $thumb.data('full');
            var imageUrl = $thumb.data('image');
            var title = $thumb.data('title');
            var altText = $thumb.data('alt');
            
            // Update active state
            $thumbnails.removeClass('active');
            $thumb.addClass('active');
            
            // Add loading state
            $mainWrapper.addClass('loading');
            
            // Preload image
            var img = new Image();
            
            img.onload = function() {
                // Update main image
                $mainImg.attr('src', imageUrl);
                $mainImg.attr('alt', altText);
                
                // Update lightbox link
                if ($mainLink.length) {
                    $mainLink.attr('href', fullUrl);
                    $mainLink.attr('title', title);
                }
                
                // Update description
                if (title && $description.length) {
                    $description.text(title).parent().show();
                } else if ($description.length) {
                    $description.parent().hide();
                }
                
                // Remove loading state
                $mainWrapper.removeClass('loading');
                
                // Scroll to main image on mobile
                if ($(window).width() <= 1024) {
                    $('html, body').animate({
                        scrollTop: $mainWrapper.offset().top - 20
                    }, 300);
                }
            };
            
            img.onerror = function() {
                $mainWrapper.removeClass('loading');
                console.error('Failed to load image:', imageUrl);
            };
            
            img.src = imageUrl;
        });
    }
    
    /**
     * Initialize keyboard navigation
     */
    function initKeyboardNav($gallery) {
        var $thumbnails = $gallery.find('.ngvf-thumb');
        
        $thumbnails.on('keydown', function(e) {
            var $current = $(this);
            var currentIndex = $thumbnails.index($current);
            var $target = null;
            
            switch(e.keyCode) {
                case 37: // Left
                case 38: // Up
                    e.preventDefault();
                    if (currentIndex > 0) {
                        $target = $thumbnails.eq(currentIndex - 1);
                    }
                    break;
                    
                case 39: // Right
                case 40: // Down
                    e.preventDefault();
                    if (currentIndex < $thumbnails.length - 1) {
                        $target = $thumbnails.eq(currentIndex + 1);
                    }
                    break;
                    
                case 36: // Home
                    e.preventDefault();
                    $target = $thumbnails.first();
                    break;
                    
                case 35: // End
                    e.preventDefault();
                    $target = $thumbnails.last();
                    break;
            }
            
            if ($target) {
                $target.focus().trigger('click');
                scrollThumbnailIntoView($target, $gallery);
            }
        });
    }
    
    /**
     * Scroll thumbnail into view
     */
    function scrollThumbnailIntoView($thumb, $gallery) {
        var $container = $gallery.find('.ngvf-thumbnails');
        
        if ($container.length === 0 || $(window).width() <= 1024) {
            return;
        }
        
        var containerTop = $container.scrollTop();
        var containerHeight = $container.height();
        var thumbTop = $thumb.position().top;
        var thumbHeight = $thumb.outerHeight();
        
        if (thumbTop < 0) {
            $container.animate({
                scrollTop: containerTop + thumbTop - 10
            }, 300);
        } else if (thumbTop + thumbHeight > containerHeight) {
            $container.animate({
                scrollTop: containerTop + thumbTop + thumbHeight - containerHeight + 10
            }, 300);
        }
    }
    
})(jQuery);
