/**
 * NGVF Lightbox JavaScript
 * 
 * @package NGVF_Standalone
 * @version 2.0.0
 */

(function($) {
    'use strict';
    
    var NGVFLightbox = {
        
        overlay: null,
        currentGallery: null,
        currentIndex: 0,
        images: [],
        
        init: function() {
            this.createLightboxHTML();
            this.bindEvents();
        },
        
        createLightboxHTML: function() {
            if ($('#ngvf-lightbox').length > 0) {
                return;
            }
            
            var html = '<div id="ngvf-lightbox" class="ngvf-lightbox-overlay">' +
                       '  <button class="ngvf-lightbox-close" aria-label="Close">&times;</button>' +
                       '  <button class="ngvf-lightbox-nav ngvf-lightbox-prev" aria-label="Previous">&lsaquo;</button>' +
                       '  <button class="ngvf-lightbox-nav ngvf-lightbox-next" aria-label="Next">&rsaquo;</button>' +
                       '  <div class="ngvf-lightbox-counter"></div>' +
                       '  <div class="ngvf-lightbox-content">' +
                       '    <div class="ngvf-lightbox-loading"></div>' +
                       '    <img src="" alt="" />' +
                       '  </div>' +
                       '</div>';
            
            $('body').append(html);
            this.overlay = $('#ngvf-lightbox');
        },
        
        bindEvents: function() {
            var self = this;
            
            // Main image click
            $(document).on('click', '.ngvf-main-link', function(e) {
                e.preventDefault();
                var $gallery = $(this).closest('.ngvf-standalone-gallery');
                self.openLightbox($gallery, 0);
            });
            
            // Thumbnail click in lightbox mode
            $(document).on('click', '.ngvf-thumb', function() {
                var $gallery = $(this).closest('.ngvf-standalone-gallery');
                var index = $gallery.find('.ngvf-thumb').index($(this));
                
                if (self.overlay && self.overlay.hasClass('active')) {
                    self.showImage(index);
                }
            });
            
            // Close button
            this.overlay.on('click', '.ngvf-lightbox-close', function(e) {
                e.stopPropagation();
                self.closeLightbox();
            });
            
            // Overlay click (outside image)
            this.overlay.on('click', function(e) {
                if (e.target === this) {
                    self.closeLightbox();
                }
            });
            
            // Navigation buttons
            this.overlay.on('click', '.ngvf-lightbox-prev', function(e) {
                e.stopPropagation();
                self.previousImage();
            });
            
            this.overlay.on('click', '.ngvf-lightbox-next', function(e) {
                e.stopPropagation();
                self.nextImage();
            });
            
            // Content click shouldn't close
            this.overlay.on('click', '.ngvf-lightbox-content', function(e) {
                e.stopPropagation();
            });
            
            // Keyboard navigation
            $(document).on('keydown', function(e) {
                if (!self.overlay.hasClass('active')) {
                    return;
                }
                
                switch(e.keyCode) {
                    case 27: // ESC
                        self.closeLightbox();
                        break;
                    case 37: // Left arrow
                        self.previousImage();
                        break;
                    case 39: // Right arrow
                        self.nextImage();
                        break;
                }
            });
        },
        
        openLightbox: function($gallery, startIndex) {
            this.currentGallery = $gallery;
            this.currentIndex = startIndex || 0;
            
            // Get all images from gallery
            this.images = [];
            $gallery.find('.ngvf-thumb').each(function() {
                var $thumb = $(this);
                self.images.push({
                    full: $thumb.data('full'),
                    alt: $thumb.data('alt'),
                    title: $thumb.data('title')
                });
            });
            
            // Get current active thumbnail index
            var $activeThumb = $gallery.find('.ngvf-thumb.active');
            if ($activeThumb.length) {
                this.currentIndex = $gallery.find('.ngvf-thumb').index($activeThumb);
            }
            
            $('body').addClass('ngvf-lightbox-open');
            this.overlay.addClass('active');
            this.showImage(this.currentIndex);
        },
        
        closeLightbox: function() {
            this.overlay.removeClass('active');
            $('body').removeClass('ngvf-lightbox-open');
            this.overlay.find('img').attr('src', '');
        },
        
        showImage: function(index) {
            if (index < 0 || index >= this.images.length) {
                return;
            }
            
            this.currentIndex = index;
            var imageData = this.images[index];
            
            var $img = this.overlay.find('.ngvf-lightbox-content img');
            var $loading = this.overlay.find('.ngvf-lightbox-loading');
            
            // Show loading
            $loading.show();
            $img.hide();
            
            // Preload image
            var newImg = new Image();
            
            newImg.onload = function() {
                $img.attr('src', imageData.full);
                $img.attr('alt', imageData.alt);
                $loading.hide();
                $img.fadeIn(200);
            };
            
            newImg.onerror = function() {
                $loading.hide();
                console.error('Failed to load image:', imageData.full);
            };
            
            newImg.src = imageData.full;
            
            // Update counter
            this.updateCounter();
            
            // Update navigation buttons
            this.updateNavButtons();
            
            // Update active thumbnail
            if (this.currentGallery) {
                this.currentGallery.find('.ngvf-thumb').removeClass('active');
                this.currentGallery.find('.ngvf-thumb').eq(index).addClass('active');
            }
        },
        
        nextImage: function() {
            if (this.currentIndex < this.images.length - 1) {
                this.showImage(this.currentIndex + 1);
            }
        },
        
        previousImage: function() {
            if (this.currentIndex > 0) {
                this.showImage(this.currentIndex - 1);
            }
        },
        
        updateCounter: function() {
            var text = (this.currentIndex + 1) + ' / ' + this.images.length;
            this.overlay.find('.ngvf-lightbox-counter').text(text);
        },
        
        updateNavButtons: function() {
            var $prev = this.overlay.find('.ngvf-lightbox-prev');
            var $next = this.overlay.find('.ngvf-lightbox-next');
            
            // Disable/enable previous button
            if (this.currentIndex === 0) {
                $prev.addClass('disabled');
            } else {
                $prev.removeClass('disabled');
            }
            
            // Disable/enable next button
            if (this.currentIndex === this.images.length - 1) {
                $next.addClass('disabled');
            } else {
                $next.removeClass('disabled');
            }
        }
    };
    
    // Make it global for the main script to use
    window.NGVFLightbox = NGVFLightbox;
    
    // Initialize when document is ready
    $(document).ready(function() {
        NGVFLightbox.init();
    });
    
    // Fix for variable reference
    var self = NGVFLightbox;
    
})(jQuery);
